from flask import Flask, request, jsonify, send_from_directory
import json
import os
import requests

# Global variable to store the current working directory
current_directory = os.getcwd()

# OpenAI API endpoint
OPENAI_API_URL = "https://api.openai.com/v1/chat/completions"

# API key management
API_KEY_FILE = "api_key.enc"
OPENAI_API_KEY = ""

# Use an absolute base directory reference so we can safely load files
BASE_DIR = os.path.dirname(os.path.abspath(__file__))

# AI Chat settings file
AI_SETTINGS_FILE = os.path.join(BASE_DIR, 'ai_settings.json')

app = Flask(__name__)

PORT = 8047

def ensure_ai_directory():
    """Ensure AI directory exists"""
    ai_dir = os.path.dirname(AI_SETTINGS_FILE)
    if not os.path.exists(ai_dir):
        os.makedirs(ai_dir, exist_ok=True)
    return ai_dir

# Serve static files
@app.route('/')
def index():
    return send_from_directory(BASE_DIR, 'index.html')

@app.route('/<path:filename>')
def static_files(filename):
    return send_from_directory(BASE_DIR, filename)

# AI Chat settings endpoints
@app.route('/api/aichat/settings', methods=['GET'])
def get_aichat_settings():
    """Get AI Chat settings"""
    try:
        full_key = request.args.get('full_key') == 'true'
        print(f"AI Settings request - full_key: {full_key}, file exists: {os.path.exists(AI_SETTINGS_FILE)}")
        
        if os.path.exists(AI_SETTINGS_FILE):
            with open(AI_SETTINGS_FILE, 'r') as f:
                settings = json.load(f)
                print(f"AI Settings loaded - has chatgpt_key: {'chatgpt_key' in settings and bool(settings.get('chatgpt_key'))}")
                print(f"AI Settings loaded - has claude_key: {'claude_key' in settings and bool(settings.get('claude_key'))}")
                print(f"AI Settings loaded - has perplexity_key: {'perplexity_key' in settings and bool(settings.get('perplexity_key'))}")
                
                # Check if this is a request for the full API key (from Desktop Assistant)
                if not full_key:
                    # Don't send the full API keys for security (UI display)
                    if 'chatgpt_key' in settings and settings['chatgpt_key']:
                        settings['chatgpt_key'] = settings['chatgpt_key'][:8] + '...' if len(settings['chatgpt_key']) > 8 else settings['chatgpt_key']
                    if 'claude_key' in settings and settings['claude_key']:
                        settings['claude_key'] = settings['claude_key'][:8] + '...' if len(settings['claude_key']) > 8 else settings['claude_key']
                    if 'perplexity_key' in settings and settings['perplexity_key']:
                        settings['perplexity_key'] = settings['perplexity_key'][:8] + '...' if len(settings['perplexity_key']) > 8 else settings['perplexity_key']
                return jsonify(settings)
        else:
            return jsonify({
                'chatgpt_key': '',
                'claude_key': '',
                'perplexity_key': '',
                'system_prompt': 'You are a helpful AI assistant running on a BlackBerry device. Be concise but informative.',
                'model': 'gpt-3.5-turbo',
                'provider': 'chatgpt'
            })
    except Exception as e:
        return jsonify({'error': f'Error loading settings: {str(e)}'}), 500

@app.route('/api/aichat/settings', methods=['POST'])
def save_aichat_settings():
    """Save AI Chat settings"""
    try:
        ensure_ai_directory()
        data = request.get_json()
        
        # Load existing settings to preserve full API keys
        existing_settings = {}
        if os.path.exists(AI_SETTINGS_FILE):
            try:
                with open(AI_SETTINGS_FILE, 'r') as f:
                    existing_settings = json.load(f)
            except:
                pass
        
        # Update settings
        settings = {
            'chatgpt_key': data.get('chatgpt_key', existing_settings.get('chatgpt_key', '')),
            'claude_key': data.get('claude_key', existing_settings.get('claude_key', '')),
            'perplexity_key': data.get('perplexity_key', existing_settings.get('perplexity_key', '')),
            'system_prompt': data.get('system_prompt', 'You are a helpful AI assistant.'),
            'model': data.get('model', existing_settings.get('model', 'gpt-3.5-turbo')),
            'provider': data.get('provider', existing_settings.get('provider', 'chatgpt'))
        }
        
        with open(AI_SETTINGS_FILE, 'w') as f:
            json.dump(settings, f, indent=2)
        
        return jsonify({'message': 'Settings saved successfully'})
    except Exception as e:
        return jsonify({'error': f'Error saving settings: {str(e)}'}), 500

@app.route('/api/aichat/send', methods=['POST'])
def send_aichat_message():
    """Send message to AI API (ChatGPT or Claude)"""
    try:
        data = request.get_json()
        message = data.get('message', '').strip()
        conversation = data.get('conversation', [])
        system_prompt = data.get('system_prompt', 'You are a helpful AI assistant.')
        model = data.get('model', 'gpt-3.5-turbo')
        provider = data.get('provider', 'chatgpt')
        
        if not message:
            return jsonify({'error': 'Message is required'}), 400
        
        # Load API keys from settings
        chatgpt_key = None
        claude_key = None
        perplexity_key = None
        if os.path.exists(AI_SETTINGS_FILE):
            try:
                with open(AI_SETTINGS_FILE, 'r') as f:
                    settings = json.load(f)
                    chatgpt_key = settings.get('chatgpt_key')
                    claude_key = settings.get('claude_key')
                    perplexity_key = settings.get('perplexity_key')
            except:
                pass
        
        # Add user message to conversation
        conversation.append({
            'role': 'user',
            'content': message
        })
        
        if provider == 'chatgpt':
            return handle_chatgpt_request(conversation, system_prompt, model, chatgpt_key)
        elif provider == 'claude':
            return handle_claude_request(conversation, system_prompt, model, claude_key)
        elif provider == 'perplexity':
            return handle_perplexity_request(conversation, system_prompt, model, perplexity_key)
        else:
            return jsonify({'error': 'Invalid provider specified'}), 400
            
    except Exception as e:
        return jsonify({'error': f'Error processing request: {str(e)}'}), 500

def handle_chatgpt_request(conversation, system_prompt, model, api_key):
    """Handle ChatGPT API request"""
    if not api_key:
        return jsonify({'error': 'ChatGPT API key not configured. Please set it in Settings.'}), 400
    
    try:
        # Prepare ChatGPT API request
        messages = [{'role': 'system', 'content': system_prompt}]
        messages.extend(conversation)
        
        chatgpt_request = {
            'model': model,
            'messages': messages,
            'max_tokens': 1000,
            'temperature': 0.7
        }
        
        # Send request to ChatGPT API
        headers = {
            'Content-Type': 'application/json',
            'Authorization': f'Bearer {api_key}'
        }
        
        response = requests.post(
            OPENAI_API_URL,
            headers=headers,
            json=chatgpt_request,
            timeout=30
        )
        
        if response.status_code == 200:
            result = response.json()
            assistant_message = result['choices'][0]['message']['content']
            
            # Add assistant response to conversation
            conversation.append({
                'role': 'assistant',
                'content': assistant_message
            })
            
            return jsonify({
                'response': assistant_message,
                'conversation': conversation
            })
        else:
            error_msg = f"ChatGPT API error: {response.status_code}"
            if response.text:
                try:
                    error_data = response.json()
                    error_msg = error_data.get('error', {}).get('message', error_msg)
                except:
                    pass
            return jsonify({'error': error_msg}), 400
            
    except requests.exceptions.Timeout:
        return jsonify({'error': 'Request timed out. Please try again.'}), 408
    except Exception as e:
        return jsonify({'error': f'ChatGPT API error: {str(e)}'}), 500

def handle_claude_request(conversation, system_prompt, model, api_key):
    """Handle Claude API request"""
    if not api_key:
        return jsonify({'error': 'Claude API key not configured. Please set it in Settings.'}), 400
    
    try:
        # Prepare Claude API request
        claude_messages = []
        for msg in conversation:
            claude_messages.append({
                'role': msg['role'],
                'content': msg['content']
            })
        
        claude_request = {
            'model': model,
            'max_tokens': 1000,
            'system': system_prompt,
            'messages': claude_messages
        }
        
        # Send request to Claude API
        headers = {
            'Content-Type': 'application/json',
            'x-api-key': api_key,
            'anthropic-version': '2023-06-01'
        }
        
        response = requests.post(
            'https://api.anthropic.com/v1/messages',
            headers=headers,
            json=claude_request,
            timeout=30
        )
        
        if response.status_code == 200:
            result = response.json()
            assistant_message = result['content'][0]['text']
            
            # Add assistant response to conversation
            conversation.append({
                'role': 'assistant',
                'content': assistant_message
            })
            
            return jsonify({
                'response': assistant_message,
                'conversation': conversation
            })
        else:
            error_msg = f"Claude API error: {response.status_code}"
            if response.text:
                try:
                    error_data = response.json()
                    error_msg = error_data.get('error', {}).get('message', error_msg)
                except:
                    pass
            return jsonify({'error': error_msg}), 400
            
    except requests.exceptions.Timeout:
        return jsonify({'error': 'Request timed out. Please try again.'}), 408
    except Exception as e:
        return jsonify({'error': f'Claude API error: {str(e)}'}), 500

def handle_perplexity_request(conversation, system_prompt, model, api_key):
    """Handle Perplexity API request"""
    if not api_key:
        return jsonify({'error': 'Perplexity API key not configured. Please set it in Settings.'}), 400
    
    try:
        # Prepare Perplexity API request - Perplexity doesn't support system messages
        # Instead, we'll prepend the system prompt to the first user message
        messages = []
        
        if conversation:
            # Add system prompt to the first user message
            first_message = conversation[0].copy()
            if first_message['role'] == 'user':
                first_message['content'] = f"{system_prompt}\n\n{first_message['content']}"
            messages.append(first_message)
            
            # Add the rest of the conversation
            messages.extend(conversation[1:])
        
        perplexity_request = {
            'model': model,
            'messages': messages,
            'max_tokens': 1000,
            'temperature': 0.7
        }
        
        # Send request to Perplexity API
        headers = {
            'Content-Type': 'application/json',
            'Authorization': f'Bearer {api_key}'
        }
        
        response = requests.post(
            'https://api.perplexity.ai/chat/completions',
            headers=headers,
            json=perplexity_request,
            timeout=30
        )
        
        if response.status_code == 200:
            result = response.json()
            assistant_message = result['choices'][0]['message']['content']
            
            # Add assistant response to conversation
            conversation.append({
                'role': 'assistant',
                'content': assistant_message
            })
            
            return jsonify({
                'response': assistant_message,
                'conversation': conversation
            })
        else:
            error_msg = f"Perplexity API error: {response.status_code}"
            if response.text:
                try:
                    error_data = response.json()
                    error_msg = error_data.get('error', {}).get('message', error_msg)
                except:
                    pass
            return jsonify({'error': error_msg}), 400
            
    except requests.exceptions.Timeout:
        return jsonify({'error': 'Request timed out. Please try again.'}), 408
    except Exception as e:
        return jsonify({'error': f'Perplexity API error: {str(e)}'}), 500

# Get available models for each provider
@app.route('/api/aichat/models/<provider>', methods=['GET'])
def get_models(provider):
    """Get available models for the specified provider"""
    if provider == 'chatgpt':
        return jsonify({
            'models': [
                'gpt-3.5-turbo',
                'gpt-4',
                'gpt-4-turbo-preview'
            ]
        })
    elif provider == 'claude':
        return jsonify({
            'models': [
                'claude-3-haiku-20240307',
                'claude-3-sonnet-20240229',
                'claude-3-opus-20240229'
            ]
        })
    elif provider == 'perplexity':
        return jsonify({
            'models': [
                'sonar',
                'sonar-pro',
                'sonar-medium',
                'sonar-small'
            ]
        })
    else:
        return jsonify({'error': 'Invalid provider'}), 400

if __name__ == "__main__":
    app.run(host="0.0.0.0", port=PORT, debug=True)