#!/usr/bin/env python3
"""
Term49 Settings Manager
BB10 Compatible Settings App for Terminal Configuration

Features:
- .term48rc font size and font type management
- .profile editing with startup services
- Desktop naming functionality
- SSH daemon auto-start with loop
- Single-file deployment compatible with taskapp.py
"""

import http.server
import socketserver
import os
import json
import urllib.parse
import re
import subprocess
import shutil
import time
from threading import Thread

PORT = 8030
BASE_DIR = os.path.dirname(__file__)

# File paths
TERM48RC_PATH = os.path.expanduser('~/.term48rc')
PROFILE_PATH = os.path.expanduser('~/.profile')
SSH_DIR = os.path.expanduser('~/.ssh')
AUTHORIZED_KEYS_PATH = os.path.expanduser('~/.ssh/authorized_keys')

# Available fonts (common BB10/QNX fonts)
AVAILABLE_FONTS = [
    "/usr/fonts/font_repository/monotype/cour.ttf",
    "/usr/fonts/font_repository/monotype/arial.ttf", 
    "/usr/fonts/font_repository/monotype/times.ttf",
    "/usr/fonts/font_repository/dejavu/DejaVuSansMono.ttf",
    "/usr/fonts/font_repository/liberation/LiberationMono-Regular.ttf"
]

# Cache for configuration data
config_cache = {}

HTML_CONTENT = '''<!DOCTYPE html>
<html>
<head>
    <meta name="viewport" content="initial-scale=1, user-scalable=no">
    <title>Term49 Settings</title>
    
    <!-- Home Screen Icon Support -->
    <link rel="icon" href="https://berrystore.sw7ft.com/apps/app-icons/term49-settings.png" type="image/png">
    <link rel="apple-touch-icon" href="https://berrystore.sw7ft.com/apps/app-icons/term49-settings.png">
    <link rel="apple-touch-icon-precomposed" href="https://berrystore.sw7ft.com/apps/app-icons/term49-settings.png">
    <meta name="apple-mobile-web-app-capable" content="yes">
    <meta name="apple-mobile-web-app-status-bar-style" content="black">
    <meta name="apple-mobile-web-app-title" content="Term49 Settings">
    <meta name="application-name" content="Term49 Settings">
    <meta name="theme-color" content="#1e1e1e">
    <style>
        body {
            background-color: #1e1e1e;
            color: white;
            font-family: "Slate Pro", Slate, "Myriad Pro", Helvetica, sans-serif;
            font-size: 14px;
            margin: 0;
            padding: 0;
            line-height: 1.4;
        }
        
        .header {
            background-color: #2b2b2b;
            border-bottom: 2px solid #00769e;
            padding: 15px 20px;
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            z-index: 1000;
            display: flex;
            justify-content: space-between;
            align-items: center;
            height: 50px;
        }
        
        h1 {
            font-size: 18px;
            margin: 0;
            color: #00769e;
        }
        
        .status-indicator {
            width: 12px;
            height: 12px;
            border-radius: 50%;
            background-color: #666;
            transition: background-color 0.3s;
        }
        
        .status-indicator.connected {
            background-color: #4CAF50;
        }
        
        .nav-tabs {
            background-color: #2b2b2b;
            border-bottom: 1px solid #333;
            padding: 0 20px;
            position: fixed;
            top: 80px;
            left: 0;
            right: 0;
            z-index: 999;
            display: flex;
            overflow-x: auto;
        }
        
        .nav-tab {
            background: none;
            border: none;
            color: #ccc;
            padding: 12px 16px;
            cursor: pointer;
            border-bottom: 2px solid transparent;
            white-space: nowrap;
            font-size: 13px;
            transition: all 0.3s;
        }
        
        .nav-tab:hover {
            color: white;
            background-color: rgba(0, 118, 158, 0.1);
        }
        
        .nav-tab.active {
            color: #00769e;
            border-bottom-color: #00769e;
        }
        
        .content {
            margin-top: 130px;
            padding: 20px;
            max-width: 800px;
            margin-left: auto;
            margin-right: auto;
        }
        
        .tab-content {
            display: none;
        }
        
        .tab-content.active {
            display: block;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-label {
            display: block;
            margin-bottom: 8px;
            color: #00769e;
            font-weight: bold;
            font-size: 13px;
        }
        
        .form-input, .form-select, .form-textarea {
            width: 100%;
            padding: 12px;
            border: 1px solid #444;
            border-radius: 4px;
            background-color: #333;
            color: white;
            font-size: 14px;
            box-sizing: border-box;
        }
        
        .form-input:focus, .form-select:focus, .form-textarea:focus {
            outline: none;
            border-color: #00769e;
            box-shadow: 0 0 0 2px rgba(0, 118, 158, 0.2);
        }
        
        .form-textarea {
            min-height: 120px;
            resize: vertical;
            font-family: "Courier New", monospace;
        }
        
        .btn {
            background-color: #00769e;
            color: white;
            border: none;
            padding: 12px 24px;
            border-radius: 4px;
            cursor: pointer;
            font-size: 14px;
            margin-right: 10px;
            margin-bottom: 10px;
            transition: background-color 0.3s;
        }
        
        .btn:hover {
            background-color: #005577;
        }
        
        .btn-secondary {
            background-color: #666;
        }
        
        .btn-secondary:hover {
            background-color: #555;
        }
        
        .btn-danger {
            background-color: #d32f2f;
        }
        
        .btn-danger:hover {
            background-color: #b71c1c;
        }
        
        .status-message {
            padding: 12px;
            border-radius: 4px;
            margin-bottom: 20px;
            display: none;
        }
        
        .status-success {
            background-color: rgba(76, 175, 80, 0.2);
            border: 1px solid #4CAF50;
            color: #4CAF50;
        }
        
        .status-error {
            background-color: rgba(211, 47, 47, 0.2);
            border: 1px solid #d32f2f;
            color: #f44336;
        }
        
        .status-info {
            background-color: rgba(0, 118, 158, 0.2);
            border: 1px solid #00769e;
            color: #00769e;
        }
        
        .config-section {
            background-color: #2b2b2b;
            border: 1px solid #444;
            border-radius: 6px;
            padding: 20px;
            margin-bottom: 20px;
        }
        
        .config-section h3 {
            margin-top: 0;
            color: #00769e;
            font-size: 16px;
            border-bottom: 1px solid #444;
            padding-bottom: 10px;
        }
        
        .font-preview {
            background-color: #1a1a1a;
            border: 1px solid #444;
            padding: 15px;
            margin-top: 10px;
            border-radius: 4px;
            font-family: monospace;
            color: #00ff00;
        }
        
        .service-item {
            background-color: #333;
            border: 1px solid #444;
            border-radius: 4px;
            padding: 15px;
            margin-bottom: 10px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .service-info {
            flex-grow: 1;
        }
        
        .service-name {
            font-weight: bold;
            color: #00769e;
        }
        
        .service-status {
            font-size: 12px;
            color: #ccc;
        }
        
        .service-actions {
            display: flex;
            gap: 10px;
        }
        
        .range-input {
            width: 100%;
            margin: 10px 0;
        }
        
        .range-value {
            display: inline-block;
            min-width: 40px;
            text-align: center;
            color: #00769e;
            font-weight: bold;
        }
        
        .desktop-name-display {
            background-color: #1a1a1a;
            border: 2px solid #00769e;
            padding: 20px;
            text-align: center;
            font-size: 24px;
            font-weight: bold;
            color: #00769e;
            border-radius: 6px;
            margin-bottom: 20px;
        }
        
        .ssh-keys-list {
            max-height: 400px;
            overflow-y: auto;
        }
        
        .ssh-key-item {
            background-color: #333;
            border: 1px solid #444;
            border-radius: 4px;
            padding: 15px;
            margin-bottom: 10px;
            word-break: break-all;
        }
        
        .ssh-key-type {
            color: #00769e;
            font-weight: bold;
            font-size: 12px;
        }
        
        .ssh-key-fingerprint {
            color: #ccc;
            font-size: 11px;
            font-family: monospace;
        }
        
        .ssh-key-comment {
            color: #999;
            font-style: italic;
        }
        
        @media (max-width: 600px) {
            .content {
                padding: 15px;
            }
            
            .nav-tab {
                padding: 10px 12px;
                font-size: 12px;
            }
            
            .service-item {
                flex-direction: column;
                align-items: flex-start;
            }
            
            .service-actions {
                margin-top: 10px;
                width: 100%;
            }
        }
    </style>
</head>
<body>
    <div class="header">
        <h1>Term49 Settings</h1>
        <div class="status-indicator" id="statusIndicator"></div>
    </div>
    
    <div class="nav-tabs">
        <button class="nav-tab active" onclick="showTab('terminal')">Terminal</button>
        <button class="nav-tab" onclick="showTab('startup')">Startup</button>
        <button class="nav-tab" onclick="showTab('desktop')">Desktop</button>
        <button class="nav-tab" onclick="showTab('services')">Services</button>
        <button class="nav-tab" onclick="showTab('sshkeys')">SSH Keys</button>
    </div>
    
    <div class="content">
        <!-- Terminal Settings Tab -->
        <div id="terminal" class="tab-content active">
            <div id="terminalStatus" class="status-message"></div>
            
            <div class="config-section">
                <h3>Font Configuration</h3>
                
                <div class="form-group">
                    <label class="form-label">Font Size:</label>
                    <input type="range" class="range-input" id="fontSizeRange" min="20" max="80" value="44" oninput="updateFontSize(this.value)">
                    <div>Size: <span class="range-value" id="fontSizeValue">44</span>px</div>
                </div>
                
                <div class="form-group">
                    <label class="form-label">Font Family:</label>
                    <select class="form-select" id="fontPath" onchange="updateFontPreview()">
                        <option value="/usr/fonts/font_repository/monotype/cour.ttf">Courier (Default)</option>
                        <option value="/usr/fonts/font_repository/monotype/arial.ttf">Arial</option>
                        <option value="/usr/fonts/font_repository/monotype/times.ttf">Times</option>
                        <option value="/usr/fonts/font_repository/dejavu/DejaVuSansMono.ttf">DejaVu Sans Mono</option>
                        <option value="/usr/fonts/font_repository/liberation/LiberationMono-Regular.ttf">Liberation Mono</option>
                    </select>
                </div>
                
                <div class="font-preview" id="fontPreview">
                    Terminal Preview Text - The quick brown fox jumps over the lazy dog<br>
                    $ ls -la<br>
                    $ python3 app.py<br>
                    Connection established...
                </div>
                
                <button class="btn" onclick="saveTerminalSettings()">Save Terminal Settings</button>
                <button class="btn btn-secondary" onclick="resetTerminalSettings()">Reset to Defaults</button>
            </div>
            
            <div class="config-section">
                <h3>Display Settings</h3>
                
                <div class="form-group">
                    <label class="form-label">Text Color (RGB):</label>
                    <input type="text" class="form-input" id="textColor" placeholder="255, 255, 255" value="255, 255, 255">
                </div>
                
                <div class="form-group">
                    <label class="form-label">Background Color (RGB):</label>
                    <input type="text" class="form-input" id="backgroundColor" placeholder="40, 0, 30" value="40, 0, 30">
                </div>
                
                <div class="form-group">
                    <label class="form-label">Screen Idle Awake:</label>
                    <select class="form-select" id="screenIdleAwake">
                        <option value="false">False</option>
                        <option value="true">True</option>
                    </select>
                </div>
            </div>
        </div>
        
        <!-- Startup Services Tab -->
        <div id="startup" class="tab-content">
            <div id="startupStatus" class="status-message"></div>
            
            <div class="config-section">
                <h3>Profile Editor</h3>
                
                <div class="form-group">
                    <label class="form-label">Profile Content (.profile):</label>
                    <textarea class="form-textarea" id="profileContent" rows="15" placeholder="Edit your .profile file content here..."></textarea>
                </div>
                
                <button class="btn" onclick="saveProfile()">Save Profile</button>
                <button class="btn btn-secondary" onclick="loadProfile()">Reload Profile</button>
                <button class="btn btn-secondary" onclick="backupProfile()">Backup Profile</button>
            </div>
        </div>
        
        <!-- Desktop Settings Tab -->
        <div id="desktop" class="tab-content">
            <div id="desktopStatus" class="status-message"></div>
            
            <div class="config-section">
                <h3>Desktop Name</h3>
                
                <div class="desktop-name-display" id="desktopNameDisplay">
                    BB10 Terminal
                </div>
                
                <div class="form-group">
                    <label class="form-label">Desktop Name:</label>
                    <input type="text" class="form-input" id="desktopName" placeholder="Enter desktop name..." value="BB10 Terminal">
                </div>
                
                <button class="btn" onclick="saveDesktopName()">Save Desktop Name</button>
                <button class="btn btn-secondary" onclick="resetDesktopName()">Reset to Default</button>
            </div>
        </div>
        
        <!-- Services Tab -->
        <div id="services" class="tab-content">
            <div id="servicesStatus" class="status-message"></div>
            
            <div class="config-section">
                <h3>SSH Daemon</h3>
                
                <div class="service-item">
                    <div class="service-info">
                        <div class="service-name">SSH Daemon (sshd)</div>
                        <div class="service-status" id="sshdStatus">Status: Checking...</div>
                    </div>
                    <div class="service-actions">
                        <button class="btn" onclick="toggleSSHD()" id="sshdToggle">Enable Auto-Start</button>
                        <button class="btn btn-secondary" onclick="checkSSHDStatus()">Check Status</button>
                    </div>
                </div>
                
                <div class="form-group">
                    <label class="form-label">SSH Configuration:</label>
                    <textarea class="form-textarea" id="sshdConfig" rows="12" placeholder="SSH daemon auto-restart configuration will be added here...">
# SSH Daemon Auto-Start with Loop
# Start SSH daemon and keep it running with automatic restart
echo "Starting SSH daemon with auto-restart loop..."
while true; do
    # Check if sshd is already running
    if ! pgrep -x "sshd" > /dev/null; then
        echo "$(date): Starting SSH daemon..."
        sshd -Dd &
        SSHD_PID=$!
        echo "SSHD started with PID: $SSHD_PID"
        
        # Wait for the process to finish
        wait $SSHD_PID
        echo "$(date): SSHD stopped (exit code: $?), restarting in 3 seconds..."
        sleep 3
    else
        echo "$(date): SSHD already running, checking again in 10 seconds..."
        sleep 10
    fi
done &
echo "SSH daemon loop started in background"
                    </textarea>
                </div>
                
                <button class="btn" onclick="saveSSHDConfig()">Save SSH Config</button>
            </div>
            
            <div class="config-section">
                <h3>Custom Startup Commands</h3>
                
                <div class="form-group">
                    <label class="form-label">Additional Startup Commands:</label>
                    <textarea class="form-textarea" id="customCommands" rows="6" placeholder="Add custom commands to run on startup..."></textarea>
                </div>
                
                <button class="btn" onclick="saveCustomCommands()">Save Custom Commands</button>
            </div>
        </div>
        
        <!-- SSH Keys Tab -->
        <div id="sshkeys" class="tab-content">
            <div id="sshkeysStatus" class="status-message"></div>
            
            <div class="config-section">
                <h3>SSH Authorized Keys</h3>
                
                <div class="form-group">
                    <label class="form-label">Add New SSH Key:</label>
                    <textarea class="form-textarea" id="newSshKey" rows="4" placeholder="Paste your SSH public key here (starts with ssh-rsa, ssh-ed25519, etc.)"></textarea>
                </div>
                
                <div class="form-group">
                    <label class="form-label">Key Comment (optional):</label>
                    <input type="text" class="form-input" id="keyComment" placeholder="e.g., user@hostname or description">
                </div>
                
                <button class="btn" onclick="addSshKey()">Add SSH Key</button>
                <button class="btn btn-secondary" onclick="loadSshKeys()">Refresh Keys</button>
                <button class="btn btn-secondary" onclick="backupSshKeys()">Backup Keys</button>
            </div>
            
            <div class="config-section">
                <h3>Current SSH Keys</h3>
                <div id="sshKeysList" class="ssh-keys-list">
                    <div class="service-item">
                        <div class="service-info">
                            <div class="service-name">Loading SSH keys...</div>
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="config-section">
                <h3>SSH Key Management</h3>
                
                <div class="form-group">
                    <label class="form-label">SSH Directory Status:</label>
                    <div id="sshDirStatus" class="service-status">Checking...</div>
                </div>
                
                <button class="btn" onclick="createSshDirectory()">Create SSH Directory</button>
                <button class="btn btn-secondary" onclick="checkSshPermissions()">Check Permissions</button>
                <button class="btn btn-danger" onclick="clearAllSshKeys()">Clear All Keys</button>
            </div>
        </div>
    </div>

    <script>
        // ES5 Compatible JavaScript for BlackBerry Passport
        var currentTab = 'terminal';
        var config = {};
        
        function showTab(tabName) {
            // Hide all tabs
            var tabs = document.querySelectorAll('.tab-content');
            for (var i = 0; i < tabs.length; i++) {
                tabs[i].classList.remove('active');
            }
            
            var navTabs = document.querySelectorAll('.nav-tab');
            for (var i = 0; i < navTabs.length; i++) {
                navTabs[i].classList.remove('active');
            }
            
            // Show selected tab
            document.getElementById(tabName).classList.add('active');
            event.target.classList.add('active');
            currentTab = tabName;
            
            // Load tab-specific data
            if (tabName === 'terminal') {
                loadTerminalSettings();
            } else if (tabName === 'startup') {
                loadProfile();
            } else if (tabName === 'desktop') {
                loadDesktopSettings();
            } else if (tabName === 'services') {
                loadServicesStatus();
            } else if (tabName === 'sshkeys') {
                loadSshKeys();
                checkSshDirectoryStatus();
            }
        }
        
        function showStatus(elementId, message, type) {
            var statusEl = document.getElementById(elementId);
            statusEl.textContent = message;
            statusEl.className = 'status-message status-' + type;
            statusEl.style.display = 'block';
            
            setTimeout(function() {
                statusEl.style.display = 'none';
            }, 5000);
        }
        
        function makeRequest(method, endpoint, data, callback) {
            var xhr = new XMLHttpRequest();
            xhr.open(method, endpoint, true);
            xhr.setRequestHeader('Content-Type', 'application/json');
            
            xhr.onreadystatechange = function() {
                if (xhr.readyState === 4) {
                    try {
                        var response = xhr.responseText ? JSON.parse(xhr.responseText) : {};
                        callback(xhr.status === 200, response);
                    } catch (e) {
                        callback(false, {error: 'Invalid response'});
                    }
                }
            };
            
            xhr.send(data ? JSON.stringify(data) : null);
        }
        
        // Terminal Settings Functions
        function loadTerminalSettings() {
            makeRequest('GET', '/api/terminal-config', null, function(success, data) {
                if (success && data.config) {
                    var config = data.config;
                    document.getElementById('fontSizeRange').value = config.font_size || 44;
                    document.getElementById('fontSizeValue').textContent = config.font_size || 44;
                    document.getElementById('fontPath').value = config.font_path || '/usr/fonts/font_repository/monotype/cour.ttf';
                    document.getElementById('textColor').value = config.text_color ? config.text_color.join(', ') : '255, 255, 255';
                    document.getElementById('backgroundColor').value = config.background_color ? config.background_color.join(', ') : '40, 0, 30';
                    document.getElementById('screenIdleAwake').value = config.screen_idle_awake ? 'true' : 'false';
                    updateFontPreview();
                }
            });
        }
        
        function updateFontSize(value) {
            document.getElementById('fontSizeValue').textContent = value;
            updateFontPreview();
        }
        
        function updateFontPreview() {
            var fontSize = document.getElementById('fontSizeRange').value;
            var fontPath = document.getElementById('fontPath').value;
            var preview = document.getElementById('fontPreview');
            
            // Simulate font preview (limited on web)
            var fontFamily = 'monospace';
            if (fontPath.indexOf('arial') !== -1) fontFamily = 'Arial, sans-serif';
            else if (fontPath.indexOf('times') !== -1) fontFamily = 'Times, serif';
            
            preview.style.fontSize = Math.max(12, fontSize * 0.3) + 'px';
            preview.style.fontFamily = fontFamily;
        }
        
        function saveTerminalSettings() {
            var textColorStr = document.getElementById('textColor').value;
            var backgroundColorStr = document.getElementById('backgroundColor').value;
            
            var textColor = textColorStr.split(',').map(function(c) { return parseInt(c.trim()); });
            var backgroundColor = backgroundColorStr.split(',').map(function(c) { return parseInt(c.trim()); });
            
            var settings = {
                font_size: parseInt(document.getElementById('fontSizeRange').value),
                font_path: document.getElementById('fontPath').value,
                text_color: textColor,
                background_color: backgroundColor,
                screen_idle_awake: document.getElementById('screenIdleAwake').value === 'true'
            };
            
            makeRequest('POST', '/api/save-terminal-config', settings, function(success, data) {
                if (success) {
                    showStatus('terminalStatus', 'Terminal settings saved successfully!', 'success');
                } else {
                    showStatus('terminalStatus', 'Error saving terminal settings: ' + (data.error || 'Unknown error'), 'error');
                }
            });
        }
        
        function resetTerminalSettings() {
            document.getElementById('fontSizeRange').value = 44;
            document.getElementById('fontSizeValue').textContent = 44;
            document.getElementById('fontPath').value = '/usr/fonts/font_repository/monotype/cour.ttf';
            document.getElementById('textColor').value = '255, 255, 255';
            document.getElementById('backgroundColor').value = '40, 0, 30';
            document.getElementById('screenIdleAwake').value = 'false';
            updateFontPreview();
        }
        
        // Profile Functions
        function loadProfile() {
            makeRequest('GET', '/api/profile', null, function(success, data) {
                if (success && data.content) {
                    document.getElementById('profileContent').value = data.content;
                } else {
                    showStatus('startupStatus', 'Error loading profile: ' + (data.error || 'Unknown error'), 'error');
                }
            });
        }
        
        function saveProfile() {
            var content = document.getElementById('profileContent').value;
            makeRequest('POST', '/api/save-profile', {content: content}, function(success, data) {
                if (success) {
                    showStatus('startupStatus', 'Profile saved successfully!', 'success');
                } else {
                    showStatus('startupStatus', 'Error saving profile: ' + (data.error || 'Unknown error'), 'error');
                }
            });
        }
        
        function backupProfile() {
            makeRequest('POST', '/api/backup-profile', null, function(success, data) {
                if (success) {
                    showStatus('startupStatus', 'Profile backed up to: ' + data.backup_path, 'success');
                } else {
                    showStatus('startupStatus', 'Error backing up profile: ' + (data.error || 'Unknown error'), 'error');
                }
            });
        }
        
        // Desktop Functions
        function loadDesktopSettings() {
            makeRequest('GET', '/api/desktop-config', null, function(success, data) {
                if (success && data.desktop_name) {
                    document.getElementById('desktopName').value = data.desktop_name;
                    document.getElementById('desktopNameDisplay').textContent = data.desktop_name;
                }
            });
        }
        
        function saveDesktopName() {
            var name = document.getElementById('desktopName').value || 'BB10 Terminal';
            makeRequest('POST', '/api/save-desktop-name', {name: name}, function(success, data) {
                if (success) {
                    document.getElementById('desktopNameDisplay').textContent = name;
                    showStatus('desktopStatus', 'Desktop name saved successfully!', 'success');
                } else {
                    showStatus('desktopStatus', 'Error saving desktop name: ' + (data.error || 'Unknown error'), 'error');
                }
            });
        }
        
        function resetDesktopName() {
            document.getElementById('desktopName').value = 'BB10 Terminal';
            document.getElementById('desktopNameDisplay').textContent = 'BB10 Terminal';
        }
        
        // Services Functions
        function loadServicesStatus() {
            checkSSHDStatus();
            loadSSHDConfig();
            loadCustomCommands();
        }
        
        function checkSSHDStatus() {
            makeRequest('GET', '/api/sshd-status', null, function(success, data) {
                var statusEl = document.getElementById('sshdStatus');
                var toggleEl = document.getElementById('sshdToggle');
                
                if (success) {
                    statusEl.textContent = 'Status: ' + (data.enabled ? 'Auto-start enabled' : 'Auto-start disabled');
                    toggleEl.textContent = data.enabled ? 'Disable Auto-Start' : 'Enable Auto-Start';
                } else {
                    statusEl.textContent = 'Status: Error checking';
                }
            });
        }
        
        function toggleSSHD() {
            makeRequest('POST', '/api/toggle-sshd', null, function(success, data) {
                if (success) {
                    showStatus('servicesStatus', 'SSHD auto-start ' + (data.enabled ? 'enabled' : 'disabled'), 'success');
                    checkSSHDStatus();
                } else {
                    showStatus('servicesStatus', 'Error toggling SSHD: ' + (data.error || 'Unknown error'), 'error');
                }
            });
        }
        
        function loadSSHDConfig() {
            makeRequest('GET', '/api/sshd-config', null, function(success, data) {
                if (success && data.config) {
                    document.getElementById('sshdConfig').value = data.config;
                }
            });
        }
        
        function saveSSHDConfig() {
            var config = document.getElementById('sshdConfig').value;
            makeRequest('POST', '/api/save-sshd-config', {config: config}, function(success, data) {
                if (success) {
                    showStatus('servicesStatus', 'SSH configuration saved successfully!', 'success');
                } else {
                    showStatus('servicesStatus', 'Error saving SSH config: ' + (data.error || 'Unknown error'), 'error');
                }
            });
        }
        
        function loadCustomCommands() {
            makeRequest('GET', '/api/custom-commands', null, function(success, data) {
                if (success && data.commands) {
                    document.getElementById('customCommands').value = data.commands;
                }
            });
        }
        
        function saveCustomCommands() {
            var commands = document.getElementById('customCommands').value;
            makeRequest('POST', '/api/save-custom-commands', {commands: commands}, function(success, data) {
                if (success) {
                    showStatus('servicesStatus', 'Custom commands saved successfully!', 'success');
                } else {
                    showStatus('servicesStatus', 'Error saving custom commands: ' + (data.error || 'Unknown error'), 'error');
                }
            });
        }
        
        // SSH Keys Functions
        function loadSshKeys() {
            makeRequest('GET', '/api/ssh-keys', null, function(success, data) {
                if (success) {
                    displaySshKeys(data.keys || []);
                } else {
                    showStatus('sshkeysStatus', 'Error loading SSH keys: ' + (data.error || 'Unknown error'), 'error');
                }
            });
        }
        
        function displaySshKeys(keys) {
            var container = document.getElementById('sshKeysList');
            
            if (keys.length === 0) {
                container.innerHTML = '<div class="service-item"><div class="service-info"><div class="service-name">No SSH keys found</div></div></div>';
                return;
            }
            
            var html = '';
            for (var i = 0; i < keys.length; i++) {
                var key = keys[i];
                html += '<div class="ssh-key-item">';
                html += '<div class="ssh-key-type">' + (key.type || 'Unknown') + '</div>';
                html += '<div style="margin: 5px 0; font-family: monospace; font-size: 12px;">' + (key.key || '') + '</div>';
                if (key.comment) {
                    html += '<div class="ssh-key-comment">' + key.comment + '</div>';
                }
                if (key.fingerprint) {
                    html += '<div class="ssh-key-fingerprint">' + key.fingerprint + '</div>';
                }
                html += '<div style="margin-top: 10px;">';
                html += '<button class="btn btn-danger" onclick="removeSshKey(' + i + ')">Remove</button>';
                html += '</div>';
                html += '</div>';
            }
            
            container.innerHTML = html;
        }
        
        function addSshKey() {
            var keyText = document.getElementById('newSshKey').value.trim();
            var comment = document.getElementById('keyComment').value.trim();
            
            if (!keyText) {
                showStatus('sshkeysStatus', 'Please enter an SSH key', 'error');
                return;
            }
            
            // Basic validation for SSH key format
            if (!keyText.match(/^(ssh-rsa|ssh-ed25519|ecdsa-sha2-nistp256|ecdsa-sha2-nistp384|ecdsa-sha2-nistp521|ssh-dss)\s+/)) {
                showStatus('sshkeysStatus', 'Invalid SSH key format. Key should start with ssh-rsa, ssh-ed25519, etc.', 'error');
                return;
            }
            
            var data = {
                key: keyText,
                comment: comment
            };
            
            makeRequest('POST', '/api/add-ssh-key', data, function(success, response) {
                if (success) {
                    showStatus('sshkeysStatus', 'SSH key added successfully!', 'success');
                    document.getElementById('newSshKey').value = '';
                    document.getElementById('keyComment').value = '';
                    loadSshKeys();
                } else {
                    showStatus('sshkeysStatus', 'Error adding SSH key: ' + (response.error || 'Unknown error'), 'error');
                }
            });
        }
        
        function removeSshKey(index) {
            if (!confirm('Are you sure you want to remove this SSH key?')) {
                return;
            }
            
            makeRequest('POST', '/api/remove-ssh-key', {index: index}, function(success, response) {
                if (success) {
                    showStatus('sshkeysStatus', 'SSH key removed successfully!', 'success');
                    loadSshKeys();
                } else {
                    showStatus('sshkeysStatus', 'Error removing SSH key: ' + (response.error || 'Unknown error'), 'error');
                }
            });
        }
        
        function backupSshKeys() {
            makeRequest('POST', '/api/backup-ssh-keys', null, function(success, response) {
                if (success) {
                    showStatus('sshkeysStatus', 'SSH keys backed up to: ' + response.backup_path, 'success');
                } else {
                    showStatus('sshkeysStatus', 'Error backing up SSH keys: ' + (response.error || 'Unknown error'), 'error');
                }
            });
        }
        
        function checkSshDirectoryStatus() {
            makeRequest('GET', '/api/ssh-directory-status', null, function(success, data) {
                var statusEl = document.getElementById('sshDirStatus');
                if (success) {
                    statusEl.textContent = data.status;
                } else {
                    statusEl.textContent = 'Error checking directory status';
                }
            });
        }
        
        function createSshDirectory() {
            makeRequest('POST', '/api/create-ssh-directory', null, function(success, response) {
                if (success) {
                    showStatus('sshkeysStatus', 'SSH directory created successfully!', 'success');
                    checkSshDirectoryStatus();
                } else {
                    showStatus('sshkeysStatus', 'Error creating SSH directory: ' + (response.error || 'Unknown error'), 'error');
                }
            });
        }
        
        function checkSshPermissions() {
            makeRequest('GET', '/api/ssh-permissions', null, function(success, data) {
                if (success) {
                    showStatus('sshkeysStatus', 'Permissions: ' + data.permissions, 'info');
                } else {
                    showStatus('sshkeysStatus', 'Error checking permissions: ' + (data.error || 'Unknown error'), 'error');
                }
            });
        }
        
        function clearAllSshKeys() {
            if (!confirm('Are you sure you want to clear ALL SSH keys? This action cannot be undone!')) {
                return;
            }
            
            makeRequest('POST', '/api/clear-ssh-keys', null, function(success, response) {
                if (success) {
                    showStatus('sshkeysStatus', 'All SSH keys cleared!', 'success');
                    loadSshKeys();
                } else {
                    showStatus('sshkeysStatus', 'Error clearing SSH keys: ' + (response.error || 'Unknown error'), 'error');
                }
            });
        }
        
        // Initialize app
        function initApp() {
            // Set status indicator
            document.getElementById('statusIndicator').classList.add('connected');
            
            // Load initial data
            loadTerminalSettings();
            
            // Update font preview initially
            updateFontPreview();
        }
        
        // Start app when page loads
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', initApp);
        } else {
            initApp();
        }
    </script>
</body>
</html>'''

class Term49SettingsHandler(http.server.BaseHTTPRequestHandler):
    
    def log_message(self, format, *args):
        # Suppress default logging for cleaner output
        pass
    
    def do_GET(self):
        if self.path == '/':
            self.serve_html()
        elif self.path == '/api/terminal-config':
            self.handle_get_terminal_config()
        elif self.path == '/api/profile':
            self.handle_get_profile()
        elif self.path == '/api/desktop-config':
            self.handle_get_desktop_config()
        elif self.path == '/api/sshd-status':
            self.handle_get_sshd_status()
        elif self.path == '/api/sshd-config':
            self.handle_get_sshd_config()
        elif self.path == '/api/custom-commands':
            self.handle_get_custom_commands()
        elif self.path == '/api/ssh-keys':
            self.handle_get_ssh_keys()
        elif self.path == '/api/ssh-directory-status':
            self.handle_get_ssh_directory_status()
        elif self.path == '/api/ssh-permissions':
            self.handle_get_ssh_permissions()
        else:
            self.send_error(404)
    
    def do_POST(self):
        content_length = int(self.headers.get('Content-Length', 0))
        post_data = self.rfile.read(content_length).decode('utf-8')
        
        try:
            data = json.loads(post_data) if post_data else {}
        except json.JSONDecodeError:
            data = {}
        
        if self.path == '/api/save-terminal-config':
            self.handle_save_terminal_config(data)
        elif self.path == '/api/save-profile':
            self.handle_save_profile(data)
        elif self.path == '/api/backup-profile':
            self.handle_backup_profile()
        elif self.path == '/api/save-desktop-name':
            self.handle_save_desktop_name(data)
        elif self.path == '/api/toggle-sshd':
            self.handle_toggle_sshd()
        elif self.path == '/api/save-sshd-config':
            self.handle_save_sshd_config(data)
        elif self.path == '/api/save-custom-commands':
            self.handle_save_custom_commands(data)
        elif self.path == '/api/add-ssh-key':
            self.handle_add_ssh_key(data)
        elif self.path == '/api/remove-ssh-key':
            self.handle_remove_ssh_key(data)
        elif self.path == '/api/backup-ssh-keys':
            self.handle_backup_ssh_keys()
        elif self.path == '/api/create-ssh-directory':
            self.handle_create_ssh_directory()
        elif self.path == '/api/clear-ssh-keys':
            self.handle_clear_ssh_keys()
        else:
            self.send_error(404)
    
    def serve_html(self):
        self.send_response(200)
        self.send_header('Content-type', 'text/html')
        self.end_headers()
        self.wfile.write(HTML_CONTENT.encode())
    
    def send_json(self, data):
        self.send_response(200)
        self.send_header('Content-type', 'application/json')
        self.end_headers()
        self.wfile.write(json.dumps(data).encode())
    
    def send_error_json(self, error_message, status_code=500):
        self.send_response(status_code)
        self.send_header('Content-type', 'application/json')
        self.end_headers()
        self.wfile.write(json.dumps({'error': error_message}).encode())
    
    # Terminal Configuration Handlers
    def handle_get_terminal_config(self):
        try:
            config = self.parse_term48rc()
            self.send_json({'config': config})
        except Exception as e:
            self.send_error_json(f'Error reading terminal config: {str(e)}')
    
    def handle_save_terminal_config(self, data):
        try:
            self.save_term48rc(data)
            self.send_json({'success': True})
        except Exception as e:
            self.send_error_json(f'Error saving terminal config: {str(e)}')
    
    def parse_term48rc(self):
        """Parse .term48rc file and return configuration dictionary"""
        config = {}
        
        if not os.path.exists(TERM48RC_PATH):
            return self.get_default_term48rc_config()
        
        try:
            with open(TERM48RC_PATH, 'r') as f:
                content = f.read()
            
            # Parse key-value pairs
            for line in content.split('\n'):
                line = line.strip()
                if line and not line.startswith('#') and '=' in line:
                    key, value = line.split('=', 1)
                    key = key.strip()
                    value = value.strip().rstrip(';')
                    
                    # Remove quotes
                    if value.startswith('"') and value.endswith('"'):
                        value = value[1:-1]
                    
                    # Parse arrays
                    if value.startswith('[') and value.endswith(']'):
                        # Parse array values
                        array_content = value[1:-1].strip()
                        if array_content:
                            config[key] = [int(x.strip()) for x in array_content.split(',') if x.strip().isdigit()]
                        else:
                            config[key] = []
                    # Parse boolean values
                    elif value.lower() in ['true', 'false']:
                        config[key] = value.lower() == 'true'
                    # Parse numeric values
                    elif value.isdigit():
                        config[key] = int(value)
                    else:
                        config[key] = value
            
            return config
            
        except Exception as e:
            print(f"Error parsing .term48rc: {e}")
            return self.get_default_term48rc_config()
    
    def get_default_term48rc_config(self):
        """Return default .term48rc configuration"""
        return {
            'font_path': '/usr/fonts/font_repository/monotype/cour.ttf',
            'font_size': 44,
            'text_color': [255, 255, 255],
            'background_color': [40, 0, 30],
            'screen_idle_awake': False,
            'auto_show_vkb': True,
            'tty_encoding': 'UTF-8'
        }
    
    def save_term48rc(self, config):
        """Save configuration to .term48rc file"""
        try:
            # Create backup
            if os.path.exists(TERM48RC_PATH):
                backup_path = TERM48RC_PATH + '.backup'
                shutil.copy2(TERM48RC_PATH, backup_path)
            
            # Read existing config to preserve other settings
            existing_config = self.parse_term48rc()
            
            # Update with new values
            existing_config.update(config)
            
            # Generate .term48rc content
            content = self.generate_term48rc_content(existing_config)
            
            # Write to file
            with open(TERM48RC_PATH, 'w') as f:
                f.write(content)
            
            print(f"Saved terminal configuration to {TERM48RC_PATH}")
            
        except Exception as e:
            raise Exception(f"Failed to save .term48rc: {str(e)}")
    
    def generate_term48rc_content(self, config):
        """Generate .term48rc file content from configuration dictionary"""
        lines = []
        
        # Font settings
        lines.append(f'font_path = "{config.get("font_path", "/usr/fonts/font_repository/monotype/cour.ttf")}";')
        lines.append(f'font_size = {config.get("font_size", 44)};')
        
        # Color settings
        text_color = config.get('text_color', [255, 255, 255])
        background_color = config.get('background_color', [40, 0, 30])
        lines.append(f'text_color = [ {", ".join(map(str, text_color))} ];')
        lines.append(f'background_color = [ {", ".join(map(str, background_color))} ];')
        
        # Other settings
        lines.append(f'screen_idle_awake = {str(config.get("screen_idle_awake", False)).lower()};')
        lines.append(f'auto_show_vkb = {str(config.get("auto_show_vkb", True)).lower()};')
        lines.append(f'tty_encoding = "{config.get("tty_encoding", "UTF-8")}";')
        
        # Preserve other settings from original file if they exist
        if os.path.exists(TERM48RC_PATH):
            try:
                with open(TERM48RC_PATH, 'r') as f:
                    original_content = f.read()
                
                # Extract settings we don't manage
                for line in original_content.split('\n'):
                    line = line.strip()
                    if line and not line.startswith('#') and '=' in line:
                        key = line.split('=', 1)[0].strip()
                        if key not in ['font_path', 'font_size', 'text_color', 'background_color', 'screen_idle_awake', 'auto_show_vkb', 'tty_encoding']:
                            lines.append(line)
            except:
                pass
        
        return '\n'.join(lines) + '\n'
    
    # Profile Handlers
    def handle_get_profile(self):
        try:
            if os.path.exists(PROFILE_PATH):
                with open(PROFILE_PATH, 'r') as f:
                    content = f.read()
                self.send_json({'content': content})
            else:
                self.send_json({'content': '# .profile file\n'})
        except Exception as e:
            self.send_error_json(f'Error reading profile: {str(e)}')
    
    def handle_save_profile(self, data):
        try:
            content = data.get('content', '')
            
            # Create backup
            if os.path.exists(PROFILE_PATH):
                backup_path = PROFILE_PATH + '.backup'
                shutil.copy2(PROFILE_PATH, backup_path)
            
            # Write new content
            with open(PROFILE_PATH, 'w') as f:
                f.write(content)
            
            self.send_json({'success': True})
        except Exception as e:
            self.send_error_json(f'Error saving profile: {str(e)}')
    
    def handle_backup_profile(self):
        try:
            if os.path.exists(PROFILE_PATH):
                timestamp = time.strftime('%Y%m%d_%H%M%S')
                backup_path = f"{PROFILE_PATH}.backup_{timestamp}"
                shutil.copy2(PROFILE_PATH, backup_path)
                self.send_json({'success': True, 'backup_path': backup_path})
            else:
                self.send_error_json('Profile file does not exist')
        except Exception as e:
            self.send_error_json(f'Error backing up profile: {str(e)}')
    
    # Desktop Configuration Handlers
    def handle_get_desktop_config(self):
        try:
            # Extract desktop name from profile
            desktop_name = self.get_desktop_name_from_profile()
            self.send_json({'desktop_name': desktop_name})
        except Exception as e:
            self.send_error_json(f'Error reading desktop config: {str(e)}')
    
    def handle_save_desktop_name(self, data):
        try:
            name = data.get('name', 'BB10 Terminal')
            self.set_desktop_name_in_profile(name)
            self.send_json({'success': True})
        except Exception as e:
            self.send_error_json(f'Error saving desktop name: {str(e)}')
    
    def get_desktop_name_from_profile(self):
        """Extract desktop name from profile file"""
        if not os.path.exists(PROFILE_PATH):
            return 'BB10 Terminal'
        
        try:
            with open(PROFILE_PATH, 'r') as f:
                content = f.read()
            
            # Look for desktop name echo command
            for line in content.split('\n'):
                if 'echo' in line and 'Desktop:' in line:
                    # Extract name from echo command
                    match = re.search(r'echo.*"Desktop:\s*([^"]*)"', line)
                    if match:
                        return match.group(1).strip()
            
            return 'BB10 Terminal'
        except:
            return 'BB10 Terminal'
    
    def set_desktop_name_in_profile(self, name):
        """Set desktop name in profile file"""
        if not os.path.exists(PROFILE_PATH):
            content = ''
        else:
            with open(PROFILE_PATH, 'r') as f:
                content = f.read()
        
        # Remove existing desktop name lines
        lines = content.split('\n')
        filtered_lines = []
        
        for line in lines:
            if not ('echo' in line and 'Desktop:' in line):
                filtered_lines.append(line)
        
        # Add new desktop name display
        desktop_display = f'''
# Desktop Name Display
echo "=================================================="
echo "Desktop: {name}"
echo "=================================================="
'''
        
        # Insert after initial setup but before other commands
        insert_index = 0
        for i, line in enumerate(filtered_lines):
            if line.strip().startswith('export') or line.strip().startswith('#'):
                insert_index = i + 1
            else:
                break
        
        filtered_lines.insert(insert_index, desktop_display.strip())
        
        # Write back to file
        with open(PROFILE_PATH, 'w') as f:
            f.write('\n'.join(filtered_lines))
    
    # SSH Daemon Handlers
    def handle_get_sshd_status(self):
        try:
            enabled = self.is_sshd_enabled_in_profile()
            self.send_json({'enabled': enabled})
        except Exception as e:
            self.send_error_json(f'Error checking SSHD status: {str(e)}')
    
    def handle_toggle_sshd(self):
        try:
            current_status = self.is_sshd_enabled_in_profile()
            new_status = not current_status
            
            if new_status:
                self.enable_sshd_in_profile()
            else:
                self.disable_sshd_in_profile()
            
            self.send_json({'enabled': new_status})
        except Exception as e:
            self.send_error_json(f'Error toggling SSHD: {str(e)}')
    
    def handle_get_sshd_config(self):
        try:
            config = self.get_sshd_config_from_profile()
            self.send_json({'config': config})
        except Exception as e:
            self.send_error_json(f'Error reading SSHD config: {str(e)}')
    
    def handle_save_sshd_config(self, data):
        try:
            config = data.get('config', '')
            self.set_sshd_config_in_profile(config)
            self.send_json({'success': True})
        except Exception as e:
            self.send_error_json(f'Error saving SSHD config: {str(e)}')
    
    def is_sshd_enabled_in_profile(self):
        """Check if SSHD auto-start is enabled in profile"""
        if not os.path.exists(PROFILE_PATH):
            return False
        
        try:
            with open(PROFILE_PATH, 'r') as f:
                content = f.read()
            
            return 'sshd -Dd' in content or 'sshd' in content
        except:
            return False
    
    def enable_sshd_in_profile(self):
        """Enable SSHD auto-start in profile"""
        if not os.path.exists(PROFILE_PATH):
            content = ''
        else:
            with open(PROFILE_PATH, 'r') as f:
                content = f.read()
        
        # Check if already enabled
        if 'sshd -Dd' in content:
            return
        
        # Add SSHD configuration with robust loop
        sshd_config = '''
# SSH Daemon Auto-Start with Loop
# Start SSH daemon and keep it running with automatic restart
echo "Starting SSH daemon with auto-restart loop..."
while true; do
    # Check if sshd is already running
    if ! pgrep -x "sshd" > /dev/null; then
        echo "$(date): Starting SSH daemon..."
        sshd -Dd &
        SSHD_PID=$!
        echo "SSHD started with PID: $SSHD_PID"
        
        # Wait for the process to finish
        wait $SSHD_PID
        echo "$(date): SSHD stopped (exit code: $?), restarting in 3 seconds..."
        sleep 3
    else
        echo "$(date): SSHD already running, checking again in 10 seconds..."
        sleep 10
    fi
done &
echo "SSH daemon loop started in background"
'''
        
        # Append to profile
        with open(PROFILE_PATH, 'a') as f:
            f.write(sshd_config)
    
    def disable_sshd_in_profile(self):
        """Disable SSHD auto-start in profile"""
        if not os.path.exists(PROFILE_PATH):
            return
        
        with open(PROFILE_PATH, 'r') as f:
            content = f.read()
        
        # Remove SSHD related lines
        lines = content.split('\n')
        filtered_lines = []
        skip_block = False
        
        for line in lines:
            if '# SSH Daemon Auto-Start' in line:
                skip_block = True
                continue
            elif skip_block and (line.strip() == '' or line.startswith('#')):
                if 'done &' in line:
                    skip_block = False
                continue
            elif skip_block:
                continue
            else:
                if 'sshd' not in line.lower():
                    filtered_lines.append(line)
        
        # Write back to file
        with open(PROFILE_PATH, 'w') as f:
            f.write('\n'.join(filtered_lines))
    
    def get_sshd_config_from_profile(self):
        """Get SSHD configuration from profile"""
        if not os.path.exists(PROFILE_PATH):
            return '''# SSH Daemon Auto-Start with Loop
# Start SSH daemon and keep it running with automatic restart
echo "Starting SSH daemon with auto-restart loop..."
while true; do
    # Check if sshd is already running
    if ! pgrep -x "sshd" > /dev/null; then
        echo "$(date): Starting SSH daemon..."
        sshd -Dd &
        SSHD_PID=$!
        echo "SSHD started with PID: $SSHD_PID"
        
        # Wait for the process to finish
        wait $SSHD_PID
        echo "$(date): SSHD stopped (exit code: $?), restarting in 3 seconds..."
        sleep 3
    else
        echo "$(date): SSHD already running, checking again in 10 seconds..."
        sleep 10
    fi
done &
echo "SSH daemon loop started in background"'''
        
        try:
            with open(PROFILE_PATH, 'r') as f:
                content = f.read()
            
            # Extract SSHD configuration block
            lines = content.split('\n')
            sshd_lines = []
            in_sshd_block = False
            
            for line in lines:
                if '# SSH Daemon Auto-Start' in line:
                    in_sshd_block = True
                    sshd_lines.append(line)
                elif in_sshd_block:
                    sshd_lines.append(line)
                    if 'done &' in line:
                        break
            
            if sshd_lines:
                return '\n'.join(sshd_lines)
            else:
                return '''# SSH Daemon Auto-Start with Loop
# Start SSH daemon and keep it running with automatic restart
echo "Starting SSH daemon with auto-restart loop..."
while true; do
    # Check if sshd is already running
    if ! pgrep -x "sshd" > /dev/null; then
        echo "$(date): Starting SSH daemon..."
        sshd -Dd &
        SSHD_PID=$!
        echo "SSHD started with PID: $SSHD_PID"
        
        # Wait for the process to finish
        wait $SSHD_PID
        echo "$(date): SSHD stopped (exit code: $?), restarting in 3 seconds..."
        sleep 3
    else
        echo "$(date): SSHD already running, checking again in 10 seconds..."
        sleep 10
    fi
done &
echo "SSH daemon loop started in background"'''
        except:
            return '''# SSH Daemon Auto-Start with Loop
# Start SSH daemon and keep it running with automatic restart
echo "Starting SSH daemon with auto-restart loop..."
while true; do
    # Check if sshd is already running
    if ! pgrep -x "sshd" > /dev/null; then
        echo "$(date): Starting SSH daemon..."
        sshd -Dd &
        SSHD_PID=$!
        echo "SSHD started with PID: $SSHD_PID"
        
        # Wait for the process to finish
        wait $SSHD_PID
        echo "$(date): SSHD stopped (exit code: $?), restarting in 3 seconds..."
        sleep 3
    else
        echo "$(date): SSHD already running, checking again in 10 seconds..."
        sleep 10
    fi
done &
echo "SSH daemon loop started in background"'''
    
    def set_sshd_config_in_profile(self, config):
        """Set SSHD configuration in profile"""
        # First disable existing SSHD
        self.disable_sshd_in_profile()
        
        # Add new configuration
        if not os.path.exists(PROFILE_PATH):
            content = ''
        else:
            with open(PROFILE_PATH, 'r') as f:
                content = f.read()
        
        # Append new SSHD config
        with open(PROFILE_PATH, 'a') as f:
            f.write('\n' + config + '\n')
    
    # Custom Commands Handlers
    def handle_get_custom_commands(self):
        try:
            commands = self.get_custom_commands_from_profile()
            self.send_json({'commands': commands})
        except Exception as e:
            self.send_error_json(f'Error reading custom commands: {str(e)}')
    
    def handle_save_custom_commands(self, data):
        try:
            commands = data.get('commands', '')
            self.set_custom_commands_in_profile(commands)
            self.send_json({'success': True})
        except Exception as e:
            self.send_error_json(f'Error saving custom commands: {str(e)}')
    
    def get_custom_commands_from_profile(self):
        """Get custom commands from profile"""
        if not os.path.exists(PROFILE_PATH):
            return ''
        
        try:
            with open(PROFILE_PATH, 'r') as f:
                content = f.read()
            
            # Look for custom commands section
            lines = content.split('\n')
            custom_lines = []
            in_custom_block = False
            
            for line in lines:
                if '# Custom Startup Commands' in line:
                    in_custom_block = True
                    continue
                elif in_custom_block and line.startswith('#'):
                    if 'End Custom' in line:
                        break
                    continue
                elif in_custom_block:
                    custom_lines.append(line)
            
            return '\n'.join(custom_lines)
        except:
            return ''
    
    def set_custom_commands_in_profile(self, commands):
        """Set custom commands in profile"""
        if not os.path.exists(PROFILE_PATH):
            content = ''
        else:
            with open(PROFILE_PATH, 'r') as f:
                content = f.read()
        
        # Remove existing custom commands section
        lines = content.split('\n')
        filtered_lines = []
        in_custom_block = False
        
        for line in lines:
            if '# Custom Startup Commands' in line:
                in_custom_block = True
                continue
            elif in_custom_block and '# End Custom' in line:
                in_custom_block = False
                continue
            elif not in_custom_block:
                filtered_lines.append(line)
        
        # Add new custom commands section
        if commands.strip():
            custom_section = f'''
# Custom Startup Commands
{commands}
# End Custom Startup Commands
'''
            filtered_lines.append(custom_section.strip())
        
        # Write back to file
        with open(PROFILE_PATH, 'w') as f:
            f.write('\n'.join(filtered_lines))
    
    # SSH Keys Handlers
    def handle_get_ssh_keys(self):
        try:
            keys = self.read_ssh_keys()
            self.send_json({'keys': keys})
        except Exception as e:
            self.send_error_json(f'Error reading SSH keys: {str(e)}')
    
    def handle_add_ssh_key(self, data):
        try:
            key = data.get('key', '').strip()
            comment = data.get('comment', '').strip()
            
            if not key:
                self.send_error_json('SSH key is required')
                return
            
            # Validate SSH key format
            if not self.validate_ssh_key(key):
                self.send_error_json('Invalid SSH key format')
                return
            
            # Add comment if provided
            if comment:
                key = f"{key} {comment}"
            
            self.add_ssh_key(key)
            self.send_json({'success': True})
        except Exception as e:
            self.send_error_json(f'Error adding SSH key: {str(e)}')
    
    def handle_remove_ssh_key(self, data):
        try:
            index = data.get('index', -1)
            if index < 0:
                self.send_error_json('Invalid key index')
                return
            
            self.remove_ssh_key_by_index(index)
            self.send_json({'success': True})
        except Exception as e:
            self.send_error_json(f'Error removing SSH key: {str(e)}')
    
    def handle_backup_ssh_keys(self):
        try:
            backup_path = self.backup_ssh_keys()
            self.send_json({'success': True, 'backup_path': backup_path})
        except Exception as e:
            self.send_error_json(f'Error backing up SSH keys: {str(e)}')
    
    def handle_create_ssh_directory(self):
        try:
            self.create_ssh_directory()
            self.send_json({'success': True})
        except Exception as e:
            self.send_error_json(f'Error creating SSH directory: {str(e)}')
    
    def handle_clear_ssh_keys(self):
        try:
            self.clear_all_ssh_keys()
            self.send_json({'success': True})
        except Exception as e:
            self.send_error_json(f'Error clearing SSH keys: {str(e)}')
    
    def handle_get_ssh_directory_status(self):
        try:
            status = self.get_ssh_directory_status()
            self.send_json({'status': status})
        except Exception as e:
            self.send_error_json(f'Error checking SSH directory: {str(e)}')
    
    def handle_get_ssh_permissions(self):
        try:
            permissions = self.get_ssh_permissions()
            self.send_json({'permissions': permissions})
        except Exception as e:
            self.send_error_json(f'Error checking SSH permissions: {str(e)}')
    
    # SSH Key Management Methods
    def read_ssh_keys(self):
        """Read and parse SSH keys from authorized_keys file"""
        if not os.path.exists(AUTHORIZED_KEYS_PATH):
            return []
        
        try:
            with open(AUTHORIZED_KEYS_PATH, 'r') as f:
                content = f.read()
            
            keys = []
            for line in content.split('\n'):
                line = line.strip()
                if line and not line.startswith('#'):
                    parts = line.split()
                    if len(parts) >= 2:
                        key_type = parts[0]
                        key_data = parts[1]
                        comment = ' '.join(parts[2:]) if len(parts) > 2 else ''
                        
                        # Generate fingerprint (simplified)
                        fingerprint = self.generate_ssh_fingerprint(key_data)
                        
                        keys.append({
                            'type': key_type,
                            'key': line,
                            'comment': comment,
                            'fingerprint': fingerprint
                        })
            
            return keys
        except Exception as e:
            print(f"Error reading SSH keys: {e}")
            return []
    
    def validate_ssh_key(self, key):
        """Validate SSH key format"""
        valid_types = ['ssh-rsa', 'ssh-ed25519', 'ecdsa-sha2-nistp256', 'ecdsa-sha2-nistp384', 'ecdsa-sha2-nistp521', 'ssh-dss']
        parts = key.split()
        
        if len(parts) < 2:
            return False
        
        return parts[0] in valid_types
    
    def add_ssh_key(self, key):
        """Add SSH key to authorized_keys file"""
        # Ensure SSH directory exists
        self.ensure_ssh_directory()
        
        # Check if key already exists
        existing_keys = self.read_ssh_keys()
        for existing_key in existing_keys:
            if existing_key['key'] == key:
                raise Exception('SSH key already exists')
        
        # Add key to file
        with open(AUTHORIZED_KEYS_PATH, 'a') as f:
            f.write(key + '\n')
        
        # Set proper permissions
        os.chmod(AUTHORIZED_KEYS_PATH, 0o600)
    
    def remove_ssh_key_by_index(self, index):
        """Remove SSH key by index"""
        keys = self.read_ssh_keys()
        if index >= len(keys):
            raise Exception('Invalid key index')
        
        # Remove the key at the specified index
        keys.pop(index)
        
        # Write remaining keys back to file
        self.write_ssh_keys(keys)
    
    def write_ssh_keys(self, keys):
        """Write SSH keys to authorized_keys file"""
        self.ensure_ssh_directory()
        
        with open(AUTHORIZED_KEYS_PATH, 'w') as f:
            for key in keys:
                f.write(key['key'] + '\n')
        
        # Set proper permissions
        os.chmod(AUTHORIZED_KEYS_PATH, 0o600)
    
    def backup_ssh_keys(self):
        """Create backup of SSH keys"""
        if not os.path.exists(AUTHORIZED_KEYS_PATH):
            raise Exception('No SSH keys file to backup')
        
        timestamp = time.strftime('%Y%m%d_%H%M%S')
        backup_path = f"{AUTHORIZED_KEYS_PATH}.backup_{timestamp}"
        shutil.copy2(AUTHORIZED_KEYS_PATH, backup_path)
        
        return backup_path
    
    def create_ssh_directory(self):
        """Create SSH directory with proper permissions"""
        if not os.path.exists(SSH_DIR):
            os.makedirs(SSH_DIR, mode=0o700)
        
        # Create authorized_keys file if it doesn't exist
        if not os.path.exists(AUTHORIZED_KEYS_PATH):
            with open(AUTHORIZED_KEYS_PATH, 'w') as f:
                f.write('')
            os.chmod(AUTHORIZED_KEYS_PATH, 0o600)
    
    def ensure_ssh_directory(self):
        """Ensure SSH directory and authorized_keys file exist"""
        self.create_ssh_directory()
    
    def clear_all_ssh_keys(self):
        """Clear all SSH keys"""
        if os.path.exists(AUTHORIZED_KEYS_PATH):
            with open(AUTHORIZED_KEYS_PATH, 'w') as f:
                f.write('')
            os.chmod(AUTHORIZED_KEYS_PATH, 0o600)
    
    def get_ssh_directory_status(self):
        """Get SSH directory status"""
        if not os.path.exists(SSH_DIR):
            return "SSH directory does not exist"
        
        if not os.path.exists(AUTHORIZED_KEYS_PATH):
            return "SSH directory exists, but authorized_keys file is missing"
        
        # Check permissions
        stat_info = os.stat(SSH_DIR)
        if stat_info.st_mode & 0o777 != 0o700:
            return "SSH directory exists but has incorrect permissions"
        
        stat_info = os.stat(AUTHORIZED_KEYS_PATH)
        if stat_info.st_mode & 0o777 != 0o600:
            return "SSH directory and authorized_keys exist but have incorrect permissions"
        
        return "SSH directory and authorized_keys are properly configured"
    
    def get_ssh_permissions(self):
        """Get SSH file permissions"""
        if not os.path.exists(SSH_DIR):
            return "SSH directory does not exist"
        
        if not os.path.exists(AUTHORIZED_KEYS_PATH):
            return "authorized_keys file does not exist"
        
        ssh_dir_perm = oct(os.stat(SSH_DIR).st_mode)[-3:]
        auth_keys_perm = oct(os.stat(AUTHORIZED_KEYS_PATH).st_mode)[-3:]
        
        return f"SSH dir: {ssh_dir_perm}, authorized_keys: {auth_keys_perm}"
    
    def generate_ssh_fingerprint(self, key_data):
        """Generate SSH key fingerprint (simplified)"""
        try:
            import hashlib
            import base64
            
            # Decode base64 key data
            key_bytes = base64.b64decode(key_data)
            
            # Generate MD5 fingerprint
            md5_hash = hashlib.md5(key_bytes).hexdigest()
            fingerprint = ':'.join([md5_hash[i:i+2] for i in range(0, len(md5_hash), 2)])
            
            return f"MD5:{fingerprint}"
        except:
            return "Unable to generate fingerprint"

class ThreadedTCPServer(socketserver.ThreadingMixIn, socketserver.TCPServer):
    allow_reuse_address = True

if __name__ == "__main__":
    print("============================================================")
    print("Term49 Settings Manager")
    print("============================================================")
    print(f"Port: {PORT}")
    print(f"Terminal Config: {TERM48RC_PATH}")
    print(f"Profile Config: {PROFILE_PATH}")
    print("Features:")
    print("- Terminal font and display settings")
    print("- Profile editing and backup")
    print("- Desktop naming")
    print("- SSH daemon auto-start")
    print("- SSH key management")
    print("- Custom startup commands")
    print("- BB10 compatible UI/UX")
    print("============================================================")
    print(f"🚀 Starting server on http://localhost:{PORT}")
    print("📱 Optimized for BlackBerry Passport")
    print("⏹️ Press Ctrl+C to stop")
    print("============================================================")
    
    try:
        with ThreadedTCPServer(("", PORT), Term49SettingsHandler) as httpd:
            httpd.serve_forever()
    except KeyboardInterrupt:
        print("\n🛑 Server stopped by user")
    except Exception as e:
        print(f"❌ Server error: {e}")
