#!/usr/bin/env python3

import http.server
import socketserver
import json
import os
import time
import threading
from datetime import datetime
import hashlib

# Configuration
PORT = 8041
CLIPBOARD_FILE = "/accounts/1000/clipboard/text.plain"
HISTORY_FILE = "clipboard_history.json"
MAX_HISTORY_ITEMS = 1000

class ClipboardMonitor:
    def __init__(self):
        self.history = []
        self.last_content = ""
        self.last_modified = 0
        self.running = False
        self.load_history()
        
    def load_history(self):
        """Load existing clipboard history from JSON file"""
        try:
            if os.path.exists(HISTORY_FILE):
                with open(HISTORY_FILE, 'r', encoding='utf-8') as f:
                    self.history = json.load(f)
                print(f"Loaded {len(self.history)} clipboard history items")
            else:
                self.history = []
                print("No existing clipboard history found")
        except Exception as e:
            print(f"Error loading clipboard history: {e}")
            self.history = []
    
    def save_history(self):
        """Save clipboard history to JSON file"""
        try:
            # Keep only the most recent items
            if len(self.history) > MAX_HISTORY_ITEMS:
                self.history = self.history[-MAX_HISTORY_ITEMS:]
            
            with open(HISTORY_FILE, 'w', encoding='utf-8') as f:
                json.dump(self.history, f, indent=2, ensure_ascii=False)
            print(f"Saved {len(self.history)} clipboard history items")
        except Exception as e:
            print(f"Error saving clipboard history: {e}")
    
    def get_clipboard_content(self):
        """Read current clipboard content from BB10 clipboard file"""
        try:
            if os.path.exists(CLIPBOARD_FILE):
                with open(CLIPBOARD_FILE, 'r', encoding='utf-8') as f:
                    content = f.read().strip()
                return content
            else:
                return ""
        except Exception as e:
            print(f"Error reading clipboard file: {e}")
            return ""
    
    def add_to_history(self, content):
        """Add new clipboard content to history"""
        if not content or content == self.last_content:
            return False
        
        # Create content hash for deduplication
        content_hash = hashlib.md5(content.encode('utf-8')).hexdigest()
        
        # Check if this content already exists in recent history
        for item in self.history[-10:]:  # Check last 10 items
            if item.get('hash') == content_hash:
                # Update timestamp of existing item
                item['last_seen'] = datetime.now().isoformat()
                self.save_history()
                return False
        
        # Add new item
        new_item = {
            'id': len(self.history) + 1,
            'content': content,
            'hash': content_hash,
            'timestamp': datetime.now().isoformat(),
            'last_seen': datetime.now().isoformat(),
            'length': len(content),
            'preview': content[:100] + ('...' if len(content) > 100 else '')
        }
        
        self.history.append(new_item)
        self.last_content = content
        self.save_history()
        
        print(f"Added clipboard item #{new_item['id']}: {new_item['preview']}")
        return True
    
    def monitor_clipboard(self):
        """Monitor clipboard file for changes"""
        print(f"Starting clipboard monitoring: {CLIPBOARD_FILE}")
        self.running = True
        
        while self.running:
            try:
                if os.path.exists(CLIPBOARD_FILE):
                    # Check if file was modified
                    current_modified = os.path.getmtime(CLIPBOARD_FILE)
                    
                    if current_modified > self.last_modified:
                        self.last_modified = current_modified
                        content = self.get_clipboard_content()
                        
                        if content:
                            self.add_to_history(content)
                
                time.sleep(1.0)  # Check every 1 second (faster for user experience)
                
            except Exception as e:
                print(f"Error monitoring clipboard: {e}")
                time.sleep(2)  # Wait longer on error
    
    def start_monitoring(self):
        """Start clipboard monitoring in background thread"""
        monitor_thread = threading.Thread(target=self.monitor_clipboard, daemon=True)
        monitor_thread.start()
        print("Clipboard monitoring started in background")
    
    def stop_monitoring(self):
        """Stop clipboard monitoring"""
        self.running = False
        print("Clipboard monitoring stopped")
    
    def get_history_json(self):
        """Get clipboard history as JSON"""
        return json.dumps({
            'total_items': len(self.history),
            'last_updated': datetime.now().isoformat(),
            'clipboard_file': CLIPBOARD_FILE,
            'history': self.history[-50:]  # Return last 50 items
        }, indent=2, ensure_ascii=False)

# Global clipboard monitor instance
clipboard_monitor = ClipboardMonitor()

class ClipboardHandler(http.server.SimpleHTTPRequestHandler):
    def do_GET(self):
        if self.path == '/':
            self.send_response(200)
            self.send_header('Content-type', 'text/html')
            self.end_headers()
            
            html_content = f'''<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>CopyClip - BlackBerry Clipboard Monitor</title>
    
    <!-- Home Screen Icon Support -->
    <link rel="icon" href="https://berrystore.sw7ft.com/apps/app-icons/copyclip.png" type="image/png">
    <link rel="apple-touch-icon" href="https://berrystore.sw7ft.com/apps/app-icons/copyclip.png">
    <link rel="apple-touch-icon-precomposed" href="https://berrystore.sw7ft.com/apps/app-icons/copyclip.png">
    <meta name="apple-mobile-web-app-capable" content="yes">
    <meta name="apple-mobile-web-app-status-bar-style" content="black">
    <meta name="apple-mobile-web-app-title" content="CopyClip">
    <meta name="application-name" content="CopyClip">
    <meta name="theme-color" content="#1a1a2e">
    <style>
        * {{
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }}
        
        body {{
            font-family: 'BB Alpha Sans', Arial, sans-serif;
            background: #000000;
            color: #ffffff;
            line-height: 1.4;
            font-size: 14px;
            margin: 0;
            padding: 0;
            height: 100vh;
            overflow: hidden;
        }}
        
        .header {{
            background: linear-gradient(135deg, #1a1a1a 0%, #333333 100%);
            padding: 15px;
            border-bottom: 2px solid #00769e;
            text-align: center;
        }}
        
        .header h1 {{
            color: #00769e;
            font-size: 24px;
            font-weight: bold;
            margin-bottom: 5px;
        }}
        
        .header .subtitle {{
            color: #cccccc;
            font-size: 12px;
        }}
        
        .status-bar {{
            background: #1a1a1a;
            padding: 10px 15px;
            border-bottom: 1px solid #333;
            display: flex;
            justify-content: space-between;
            align-items: center;
            flex-wrap: wrap;
        }}
        
        .status-item {{
            color: #00769e;
            font-size: 11px;
            margin: 2px 0;
        }}
        
        .controls {{
            background: #1a1a1a;
            padding: 12px;
            border-bottom: 1px solid #333;
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 8px;
            position: sticky;
            top: 0;
            z-index: 100;
        }}
        
        .btn {{
            background: #00769e;
            color: white;
            border: none;
            padding: 12px 8px;
            border-radius: 6px;
            cursor: pointer;
            font-size: 11px;
            font-weight: bold;
            text-align: center;
            transition: all 0.2s ease;
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
        }}
        
        .btn:hover {{
            background: #005577;
        }}
        
        .btn:active {{
            background: #003344;
        }}
        
        .btn.secondary {{
            background: #666666;
        }}
        
        .btn.secondary:hover {{
            background: #555555;
        }}
        
        .main-container {{
            display: flex;
            flex-direction: column;
            height: 100vh;
        }}
        
        .clipboard-list {{
            flex: 1;
            padding: 12px;
            overflow-y: auto;
            -webkit-overflow-scrolling: touch;
            scroll-behavior: smooth;
        }}
        
        .clipboard-list::-webkit-scrollbar {{
            width: 6px;
        }}
        
        .clipboard-list::-webkit-scrollbar-track {{
            background: #1a1a1a;
        }}
        
        .clipboard-list::-webkit-scrollbar-thumb {{
            background: #00769e;
            border-radius: 3px;
        }}
        
        .clipboard-list::-webkit-scrollbar-thumb:hover {{
            background: #005577;
        }}
        
        .clipboard-item {{
            background: #1a1a1a;
            border: 1px solid #333;
            border-radius: 8px;
            margin-bottom: 8px;
            padding: 14px;
            position: relative;
            cursor: pointer;
            transition: all 0.15s ease;
            user-select: none;
            -webkit-user-select: none;
            -webkit-tap-highlight-color: transparent;
        }}
        
        .clipboard-item:hover {{
            border-color: #00769e;
            background: #222222;
            box-shadow: 0 2px 12px rgba(0, 118, 158, 0.15);
        }}
        
        .clipboard-item:active {{
            transform: scale(0.98);
            background: #2a2a2a;
        }}
        
        .clipboard-item.copying {{
            border-color: #00ff00;
            background: #002200;
            animation: copyFlash 0.5s ease;
        }}
        
        @keyframes copyFlash {{
            0% {{ background: #002200; }}
            50% {{ background: #004400; }}
            100% {{ background: #002200; }}
        }}
        
        .item-header {{
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 8px;
            font-size: 11px;
            color: #888;
        }}
        
        .item-id {{
            color: #00769e;
            font-weight: bold;
        }}
        
        .item-timestamp {{
            color: #666;
        }}
        
        .item-content {{
            color: #ffffff;
            font-size: 13px;
            line-height: 1.5;
            word-wrap: break-word;
            white-space: pre-wrap;
            max-height: 120px;
            overflow-y: auto;
            background: #0a0a0a;
            padding: 10px;
            border-radius: 6px;
            border: 1px solid #333;
            margin: 8px 0;
            -webkit-overflow-scrolling: touch;
        }}
        
        .item-content::-webkit-scrollbar {{
            width: 4px;
        }}
        
        .item-content::-webkit-scrollbar-track {{
            background: #0a0a0a;
        }}
        
        .item-content::-webkit-scrollbar-thumb {{
            background: #555;
            border-radius: 2px;
        }}
        
        .item-meta {{
            margin-top: 8px;
            font-size: 10px;
            color: #666;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }}
        
        .copy-btn {{
            background: #00769e;
            color: white;
            border: none;
            padding: 6px 10px;
            border-radius: 4px;
            cursor: pointer;
            font-size: 10px;
            font-weight: bold;
            opacity: 0.9;
            transition: all 0.2s ease;
            min-width: 32px;
            height: 28px;
            display: flex;
            align-items: center;
            justify-content: center;
        }}
        
        .copy-btn:hover {{
            opacity: 1;
            background: #005577;
        }}
        
        .copy-btn:active {{
            background: #003344;
        }}
        
        .copy-tooltip {{
            position: absolute;
            top: -30px;
            right: 10px;
            background: #333;
            color: white;
            padding: 4px 8px;
            border-radius: 4px;
            font-size: 10px;
            opacity: 0;
            pointer-events: none;
            transition: opacity 0.2s ease;
            z-index: 1000;
        }}
        
        .copy-tooltip.show {{
            opacity: 1;
        }}
        
        .click-to-copy-hint {{
            position: absolute;
            top: 5px;
            right: 5px;
            font-size: 9px;
            color: #666;
            opacity: 0;
            transition: opacity 0.2s ease;
        }}
        
        .clipboard-item:hover .click-to-copy-hint {{
            opacity: 1;
        }}
        
        .empty-state {{
            text-align: center;
            padding: 40px 20px;
            color: #666;
        }}
        
        .empty-state h3 {{
            color: #00769e;
            margin-bottom: 10px;
        }}
        
        .loading {{
            text-align: center;
            padding: 20px;
            color: #00769e;
        }}
        
        /* BlackBerry Passport optimizations */
        @media (max-width: 500px) {{
            .header {{
                padding: 12px;
            }}
            
            .header h1 {{
                font-size: 20px;
            }}
            
            .header .subtitle {{
                font-size: 11px;
            }}
            
            .status-bar {{
                padding: 8px 12px;
                flex-direction: column;
                align-items: stretch;
            }}
            
            .status-item {{
                font-size: 10px;
                margin: 2px 0;
                text-align: left;
            }}
            
            .controls {{
                padding: 10px;
                gap: 6px;
                grid-template-columns: 1fr 1fr;
            }}
            
            .btn {{
                padding: 10px 6px;
                font-size: 10px;
            }}
            
            .clipboard-list {{
                padding: 8px;
            }}
            
            .clipboard-item {{
                padding: 12px;
                margin-bottom: 6px;
            }}
            
            .item-header {{
                margin-bottom: 6px;
            }}
            
            .item-content {{
                font-size: 12px;
                max-height: 100px;
                padding: 8px;
                margin: 6px 0;
            }}
            
            .item-meta {{
                font-size: 9px;
            }}
            
            .copy-btn {{
                padding: 4px 8px;
                font-size: 9px;
                min-width: 28px;
                height: 24px;
            }}
            
            .click-to-copy-hint {{
                font-size: 8px;
            }}
        }}
        
        /* Very small screens - single column buttons */
        @media (max-width: 360px) {{
            .controls {{
                grid-template-columns: 1fr;
            }}
        }}
    </style>
</head>
<body>
    <div class="main-container">
        <div class="header">
            <h1>📋 CopyClip</h1>
            <div class="subtitle">BlackBerry Clipboard Monitor</div>
        </div>
        
        <div class="status-bar">
            <div class="status-item">📁 Monitoring: {CLIPBOARD_FILE}</div>
            <div class="status-item">📊 Total Items: <span id="totalItems">0</span></div>
            <div class="status-item">🕒 Last Update: <span id="lastUpdate">--</span></div>
        </div>
        
        <div class="controls">
            <button class="btn" onclick="refreshHistory()">🔄 Refresh</button>
            <button class="btn secondary" onclick="clearHistory()">🗑️ Clear</button>
            <button class="btn secondary" onclick="exportHistory()">💾 Export</button>
            <button class="btn secondary" onclick="toggleAutoRefresh()">⏱️ <span id="autoRefreshStatus">ON</span></button>
        </div>
        
        <div class="clipboard-list" id="clipboardList">
            <div class="loading">Loading clipboard history...</div>
        </div>
    </div>

    <script>
        var clipboardHistory = [];
        var autoRefresh = true;
        var refreshInterval;
        
        // ES5 compatible date formatting
        function formatDate(dateString) {{
            var date = new Date(dateString);
            var hours = date.getHours();
            var minutes = date.getMinutes();
            var seconds = date.getSeconds();
            
            // Pad with zeros
            hours = (hours < 10 ? '0' : '') + hours;
            minutes = (minutes < 10 ? '0' : '') + minutes;
            seconds = (seconds < 10 ? '0' : '') + seconds;
            
            var day = date.getDate();
            var month = date.getMonth() + 1;
            var year = date.getFullYear();
            
            day = (day < 10 ? '0' : '') + day;
            month = (month < 10 ? '0' : '') + month;
            
            return month + '/' + day + '/' + year + ' ' + hours + ':' + minutes + ':' + seconds;
        }}
        
        function escapeHtml(text) {{
            var div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }}
        
        function refreshHistory() {{
            console.log('Refreshing clipboard history...');
            
            // Use XMLHttpRequest for ES5 compatibility
            var xhr = new XMLHttpRequest();
            xhr.open('GET', '/api/history', true);
            xhr.onreadystatechange = function() {{
                if (xhr.readyState === 4) {{
                    if (xhr.status === 200) {{
                        try {{
                            var data = JSON.parse(xhr.responseText);
                            clipboardHistory = data.history || [];
                            updateDisplay(data);
                            console.log('History refreshed:', clipboardHistory.length, 'items');
                        }} catch (e) {{
                            console.error('Error parsing history data:', e);
                            showError('Error loading clipboard history');
                        }}
                    }} else {{
                        console.error('Error loading history:', xhr.status);
                        showError('Failed to load clipboard history');
                    }}
                }}
            }};
            xhr.send();
        }}
        
        function updateDisplay(data) {{
            var listElement = document.getElementById('clipboardList');
            var totalItemsElement = document.getElementById('totalItems');
            var lastUpdateElement = document.getElementById('lastUpdate');
            
            // Update status only if changed (performance optimization)
            var newTotal = data.total_items || 0;
            var newUpdate = formatDate(data.last_updated || new Date().toISOString());
            
            if (totalItemsElement.textContent !== newTotal.toString()) {{
                totalItemsElement.textContent = newTotal;
            }}
            if (lastUpdateElement.textContent !== newUpdate) {{
                lastUpdateElement.textContent = newUpdate;
            }}
            
            if (clipboardHistory.length === 0) {{
                listElement.innerHTML = 
                    '<div class="empty-state">' +
                    '<h3>📋 No Clipboard History</h3>' +
                    '<p>Copy some text to see it appear here!</p>' +
                    '<p><strong>Monitoring:</strong> {CLIPBOARD_FILE}</p>' +
                    '<br>' +
                    '<p><em>💡 Tip: Once clipboard items appear, click any item to copy it back to your clipboard!</em></p>' +
                    '</div>';
                return;
            }}
            
            // Only rebuild if history actually changed (performance optimization)
            var currentHistoryLength = clipboardHistory.length;
            if (listElement.getAttribute('data-history-length') === currentHistoryLength.toString()) {{
                return; // No changes, skip rebuild
            }}
            listElement.setAttribute('data-history-length', currentHistoryLength);
            
            var htmlParts = [];
            // Reverse to show newest first, limit to 20 items for smooth scrolling
            var reversedHistory = clipboardHistory.slice().reverse().slice(0, 20);
            
            for (var i = 0; i < reversedHistory.length; i++) {{
                var item = reversedHistory[i];
                var escapedContent = escapeHtml(item.content);
                var quotedContent = escapedContent.replace(/"/g, '&quot;');
                
                htmlParts.push(
                    '<div class="clipboard-item" onclick="copyToClipboard(' + item.id + ', this)" data-content="' + quotedContent + '">',
                    '<div class="click-to-copy-hint">Click to copy</div>',
                    '<div class="copy-tooltip">Copied!</div>',
                    '<div class="item-header">',
                    '<span class="item-id">#' + item.id + '</span>',
                    '<span class="item-timestamp">' + formatDate(item.timestamp) + '</span>',
                    '</div>',
                    '<div class="item-content">' + escapedContent + '</div>',
                    '<div class="item-meta">',
                    '<span>Length: ' + item.length + ' chars</span>',
                    '<button class="copy-btn" onclick="event.stopPropagation(); copyToClipboard(' + item.id + ', this.parentElement.parentElement);">📋</button>'
                );
                
                if (item.last_seen !== item.timestamp) {{
                    htmlParts.push('<span>Last seen: ' + formatDate(item.last_seen) + '</span>');
                }}
                
                htmlParts.push('</div>', '</div>');
            }}
            
            var html = htmlParts.join('');
            
            // Check if we're adding new items (scroll to top smoothly)
            var wasAtTop = listElement.scrollTop < 50;
            
            listElement.innerHTML = html;
            
            // If we were at the top and have new items, stay at top
            if (wasAtTop && clipboardHistory.length > 0) {{
                listElement.scrollTop = 0;
            }}
        }}
        
        function showError(message) {{
            var listElement = document.getElementById('clipboardList');
            listElement.innerHTML = 
                '<div class="empty-state">' +
                '<h3>Error</h3>' +
                '<p>' + escapeHtml(message) + '</p>' +
                '</div>';
        }}
        
        function copyToClipboard(itemId, element) {{
            console.log('Copying item #' + itemId + ' to clipboard');
            
            // Get the content from the data attribute
            var content = element.getAttribute('data-content');
            if (!content) {{
                console.error('No content found for item #' + itemId);
                return;
            }}
            
            // Decode HTML entities
            var textarea = document.createElement('textarea');
            textarea.innerHTML = content;
            content = textarea.value;
            
            // Try modern clipboard API first
            if (navigator.clipboard && navigator.clipboard.writeText) {{
                navigator.clipboard.writeText(content).then(function() {{
                    showCopySuccess(element, 'Copied with Clipboard API!');
                }}).catch(function(err) {{
                    console.log('Clipboard API failed, trying fallback:', err);
                    fallbackCopyToClipboard(content, element);
                }});
            }} else {{
                // Fallback for older browsers (like BB10 WebKit)
                fallbackCopyToClipboard(content, element);
            }}
        }}
        
        function fallbackCopyToClipboard(text, element) {{
            console.log('Using fallback copy method');
            
            // Create temporary textarea
            var textarea = document.createElement('textarea');
            textarea.value = text;
            textarea.style.position = 'fixed';
            textarea.style.left = '-9999px';
            textarea.style.top = '-9999px';
            document.body.appendChild(textarea);
            
            try {{
                // Select and copy
                textarea.select();
                textarea.setSelectionRange(0, 99999); // For mobile devices
                
                var successful = document.execCommand('copy');
                if (successful) {{
                    showCopySuccess(element, 'Copied!');
                }} else {{
                    showCopyError(element, 'Copy failed');
                }}
            }} catch (err) {{
                console.error('Fallback copy failed:', err);
                showCopyError(element, 'Copy not supported');
            }} finally {{
                document.body.removeChild(textarea);
            }}
        }}
        
        function showCopySuccess(element, message) {{
            // Add copying class for animation
            element.classList.add('copying');
            
            // Show tooltip
            var tooltip = element.querySelector('.copy-tooltip');
            if (tooltip) {{
                tooltip.textContent = message || 'Copied!';
                tooltip.classList.add('show');
                
                // Hide tooltip after 2 seconds
                setTimeout(function() {{
                    tooltip.classList.remove('show');
                }}, 2000);
            }}
            
            // Remove copying class after animation
            setTimeout(function() {{
                element.classList.remove('copying');
            }}, 500);
            
            console.log('Copy successful:', message);
        }}
        
        function showCopyError(element, message) {{
            var tooltip = element.querySelector('.copy-tooltip');
            if (tooltip) {{
                tooltip.textContent = message || 'Copy failed';
                tooltip.style.background = '#cc3333';
                tooltip.classList.add('show');
                
                // Hide tooltip and reset color after 3 seconds
                setTimeout(function() {{
                    tooltip.classList.remove('show');
                    setTimeout(function() {{
                        tooltip.style.background = '#333';
                        tooltip.textContent = 'Copied!';
                    }}, 300);
                }}, 3000);
            }}
            
            console.error('Copy failed:', message);
        }}
        
        function clearHistory() {{
            if (confirm('Are you sure you want to clear all clipboard history?')) {{
                var xhr = new XMLHttpRequest();
                xhr.open('POST', '/api/clear', true);
                xhr.onreadystatechange = function() {{
                    if (xhr.readyState === 4) {{
                        if (xhr.status === 200) {{
                            refreshHistory();
                            console.log('History cleared');
                        }} else {{
                            alert('Failed to clear history');
                        }}
                    }}
                }};
                xhr.send();
            }}
        }}
        
        function exportHistory() {{
            var xhr = new XMLHttpRequest();
            xhr.open('GET', '/api/export', true);
            xhr.onreadystatechange = function() {{
                if (xhr.readyState === 4) {{
                    if (xhr.status === 200) {{
                        // Create download link
                        var blob = new Blob([xhr.responseText], {{ type: 'application/json' }});
                        var url = window.URL.createObjectURL(blob);
                        var a = document.createElement('a');
                        a.href = url;
                        a.download = 'clipboard_history_' + new Date().getTime() + '.json';
                        document.body.appendChild(a);
                        a.click();
                        document.body.removeChild(a);
                        window.URL.revokeObjectURL(url);
                    }} else {{
                        alert('Failed to export history');
                    }}
                }}
            }};
            xhr.send();
        }}
        
        function toggleAutoRefresh() {{
            autoRefresh = !autoRefresh;
            var statusElement = document.getElementById('autoRefreshStatus');
            statusElement.textContent = autoRefresh ? 'ON' : 'OFF';
            
            if (autoRefresh) {{
                startAutoRefresh();
            }} else {{
                stopAutoRefresh();
            }}
        }}
        
        function startAutoRefresh() {{
            if (refreshInterval) {{
                clearInterval(refreshInterval);
            }}
            refreshInterval = setInterval(refreshHistory, 3000); // Refresh every 3 seconds (less aggressive)
        }}
        
        function stopAutoRefresh() {{
            if (refreshInterval) {{
                clearInterval(refreshInterval);
                refreshInterval = null;
            }}
        }}
        
        // Initialize app
        function initApp() {{
            console.log('Initializing CopyClip...');
            refreshHistory();
            startAutoRefresh();
        }}
        
        // Start app when DOM is ready
        if (document.readyState === 'loading') {{
            document.addEventListener('DOMContentLoaded', initApp);
        }} else {{
            initApp();
        }}
    </script>
</body>
</html>'''
            
            self.wfile.write(html_content.encode('utf-8'))
            
        elif self.path == '/api/history':
            self.send_response(200)
            self.send_header('Content-type', 'application/json')
            self.send_header('Access-Control-Allow-Origin', '*')
            self.end_headers()
            
            history_json = clipboard_monitor.get_history_json()
            self.wfile.write(history_json.encode('utf-8'))
            
        elif self.path == '/api/export':
            self.send_response(200)
            self.send_header('Content-type', 'application/json')
            self.send_header('Content-Disposition', 'attachment; filename="clipboard_history.json"')
            self.send_header('Access-Control-Allow-Origin', '*')
            self.end_headers()
            
            full_history = json.dumps({
                'exported_at': datetime.now().isoformat(),
                'total_items': len(clipboard_monitor.history),
                'clipboard_file': CLIPBOARD_FILE,
                'history': clipboard_monitor.history
            }, indent=2, ensure_ascii=False)
            
            self.wfile.write(full_history.encode('utf-8'))
            
        else:
            self.send_error(404)
    
    def do_POST(self):
        if self.path == '/api/clear':
            clipboard_monitor.history = []
            clipboard_monitor.save_history()
            
            self.send_response(200)
            self.send_header('Content-type', 'application/json')
            self.send_header('Access-Control-Allow-Origin', '*')
            self.end_headers()
            
            response = json.dumps({{'status': 'success', 'message': 'History cleared'}})
            self.wfile.write(response.encode('utf-8'))
        else:
            self.send_error(404)

def main():
    print("=" * 60)
    print("🔥 CopyClip - BlackBerry Clipboard Monitor")
    print("=" * 60)
    print(f"📋 Monitoring clipboard file: {CLIPBOARD_FILE}")
    print(f"💾 History file: {HISTORY_FILE}")
    print(f"🌐 Web interface: http://localhost:{PORT}")
    print(f"📱 BlackBerry access: http://192.168.1.108:{PORT}")
    print("=" * 60)
    
    # Start clipboard monitoring
    clipboard_monitor.start_monitoring()
    
    # Start web server
    try:
        with socketserver.TCPServer(("", PORT), ClipboardHandler) as httpd:
            print(f"✅ CopyClip server running on port {PORT}")
            print("📋 Copy some text to see it appear in the history!")
            print("🔄 Press Ctrl+C to stop")
            print()
            
            httpd.serve_forever()
            
    except KeyboardInterrupt:
        print("\n🛑 Stopping CopyClip...")
        clipboard_monitor.stop_monitoring()
        print("✅ CopyClip stopped")
    except Exception as e:
        print(f"❌ Error starting server: {e}")

if __name__ == "__main__":
    main()
