#!/bin/sh
# BOCHS COMPREHENSIVE ARM ALIGNMENT FIX
# This version fixes ALL memory alignment issues that cause SIGBUS errors on ARM

# Get the directory where this script is located
SCRIPT_DIR="$(cd "$(dirname "$0")" && pwd)"

# Set library path
export LD_LIBRARY_PATH="$SCRIPT_DIR/lib:/usr/local/lib:/usr/lib:/lib:$LD_LIBRARY_PATH"

# Set BOCHS share path
export BXSHARE="$SCRIPT_DIR/share/bochs"

# Set display to XSDL server
export DISPLAY=192.168.2.2:1

echo "=========================================="
echo "BOCHS COMPREHENSIVE ARM ALIGNMENT FIX"
echo "=========================================="
echo ""
echo "🔧 ALL ARM ALIGNMENT ISSUES FIXED:"
echo "   ✅ system_read_qword  - 8-byte alignment"
echo "   ✅ system_read_dword  - 4-byte alignment" 
echo "   ✅ system_read_word   - 2-byte alignment"
echo "   ✅ system_write_qword - 8-byte alignment"
echo "   ✅ system_write_dword - 4-byte alignment"
echo "   ✅ system_write_word  - 2-byte alignment"
echo ""
echo "🎯 PREVIOUS FIXES INCLUDED:"
echo "   ✅ time0 parameter bug - SOLVED"
echo "   ✅ C++ runtime libraries - INCLUDED"
echo "   ✅ X11 GUI connection - WORKING"
echo ""
echo "This should eliminate ALL SIGBUS errors!"
echo "=========================================="
echo ""

# Create comprehensive configuration
cat > "$SCRIPT_DIR/comprehensive.bochsrc" << 'EOFCONFIG'
# BOCHS Comprehensive ARM Fix Configuration
# All alignment and compatibility issues resolved!

# Memory settings
megs: 64

# BIOS files
romimage: file=$BXSHARE/BIOS-bochs-latest
vgaromimage: file=$BXSHARE/VGABIOS-lgpl-latest

# Boot from floppy
boot: floppy
floppya: 1_44=/dev/null, status=ejected

# Display settings for X11
display_library: x

# CPU settings
cpu: count=1, ips=4000000, model=bx_generic, reset_on_triple_fault=1

# Clock settings (working!)
clock: sync=realtime

# I/O settings
mouse: enabled=0
keyboard: keymap=

# Logging
log: bochs.log
logprefix: %t%e%d
panic: action=ask
error: action=report
info: action=report
debug: action=ignore

# Display settings
private_colormap: enabled=0
EOFCONFIG

echo "=== Comprehensive ARM-Fixed Configuration Created ==="
cat "$SCRIPT_DIR/comprehensive.bochsrc"
echo ""
echo "=== Starting BOCHS (ALL ARM issues fixed!) ==="

# Run BOCHS with the comprehensively fixed binary
"$SCRIPT_DIR/bin/bochs" -q -f "$SCRIPT_DIR/comprehensive.bochsrc"
