#!/bin/sh
# Smart BOCHS XP Boot - Automatically detects XP.IMG location

# Get the directory where this script is located
SCRIPT_DIR="$(cd "$(dirname "$0")" && pwd)"

# Set library path
export LD_LIBRARY_PATH="$SCRIPT_DIR/lib:/usr/local/lib:/usr/lib:/lib:$LD_LIBRARY_PATH"

# Set BOCHS share path
export BXSHARE="$SCRIPT_DIR/share/bochs"

# Set display to XSDL server
export DISPLAY=192.168.2.2:1

echo "=========================================="
echo "SMART BOCHS XP BOOT"
echo "=========================================="

# Check for XP.IMG in common locations
XP_PATHS=(
    "/accounts/1000/removable/sdcard/BB10/XP.IMG"
    "/accounts/1000/shared/XP.IMG"
    "/accounts/1000/XP.IMG"
    "./XP.IMG"
)

XP_IMAGE=""
for path in "${XP_PATHS[@]}"; do
    if [ -f "$path" ]; then
        XP_IMAGE="$path"
        echo "✅ Found XP.IMG at: $path"
        break
    fi
done

if [ -z "$XP_IMAGE" ]; then
    echo "❌ XP.IMG not found in common locations:"
    for path in "${XP_PATHS[@]}"; do
        echo "   - $path"
    done
    echo ""
    echo "Please ensure XP.IMG is available and try again."
    echo "Current directory contents:"
    ls -la /accounts/1000/removable/sdcard/BB10/ 2>/dev/null || echo "SD card not accessible"
    exit 1
fi

# Get file size for memory optimization
XP_SIZE=$(ls -lh "$XP_IMAGE" | awk '{print $5}')
echo "📁 XP.IMG size: $XP_SIZE"

# Determine optimal memory based on image size
if [ -f "$XP_IMAGE" ]; then
    SIZE_BYTES=$(stat -c%s "$XP_IMAGE" 2>/dev/null || echo "0")
    if [ "$SIZE_BYTES" -gt 2147483648 ]; then  # > 2GB
        MEMORY=1024
        echo "🧠 Using 1GB RAM (large image detected)"
    elif [ "$SIZE_BYTES" -gt 1073741824 ]; then  # > 1GB
        MEMORY=768
        echo "🧠 Using 768MB RAM (medium image detected)"
    else
        MEMORY=512
        echo "🧠 Using 512MB RAM (standard configuration)"
    fi
else
    MEMORY=512
fi

echo ""
echo "🖱️  Mouse: Enabled with XSDL touch support"
echo "⌨️  Keyboard: Full keyboard mapping"
echo "🎮 XSDL Controls:"
echo "   - Touch = Left click"
echo "   - Long press = Right click"
echo "   - Ctrl+Middle = Toggle mouse capture"
echo "=========================================="
echo ""

# Create dynamic XP configuration
cat > "$SCRIPT_DIR/smart_xp.bochsrc" << EOFCONFIG
# Smart BOCHS Configuration for Windows XP
# Auto-configured for: $XP_IMAGE

# Memory settings
megs: $MEMORY

# BIOS files
romimage: file=\$BXSHARE/BIOS-bochs-latest
vgaromimage: file=\$BXSHARE/VGABIOS-lgpl-latest

# Boot from hard disk
boot: disk
ata0-master: type=disk, path="$XP_IMAGE", mode=flat

# Display settings for XSDL
display_library: x

# CPU settings
cpu: count=1, ips=50000000, model=bx_generic, reset_on_triple_fault=1

# Clock settings
clock: sync=realtime

# Mouse - ENABLED with proper XSDL support
mouse: enabled=1, type=ps2, toggle=ctrl+mbutton

# Keyboard - full support
keyboard: keymap=, user_shortcut=none

# Logging
log: bochs.log
logprefix: %t%e%d
panic: action=ask
error: action=report
info: action=ignore
debug: action=ignore

# Display optimization
private_colormap: enabled=0
vga: extension=vbe, update_freq=60
EOFCONFIG

echo "=== Smart XP Configuration Created ==="
echo "Memory: ${MEMORY}MB"
echo "Image: $XP_IMAGE"
echo ""
echo "=== Starting Windows XP in BOCHS ==="
echo "First boot may take 5-10 minutes..."
echo ""

# Run BOCHS
"$SCRIPT_DIR/bin/bochs" -q -f "$SCRIPT_DIR/smart_xp.bochsrc"
