#!/bin/sh
# IMG File Unlock and Reset Utility for BOCHS

echo "=========================================="
echo "IMG FILE UNLOCK & RESET UTILITY"
echo "=========================================="
echo ""

# Check for XP.IMG in common locations
XP_PATHS=(
    "/accounts/1000/removable/sdcard/BB10/XP.IMG"
    "/accounts/1000/shared/XP.IMG"
    "/accounts/1000/XP.IMG"
    "./XP.IMG"
)

XP_IMAGE=""
for path in "${XP_PATHS[@]}"; do
    if [ -f "$path" ]; then
        XP_IMAGE="$path"
        echo "📁 Found IMG file: $path"
        break
    fi
done

if [ -z "$XP_IMAGE" ]; then
    echo "❌ No IMG file found in common locations"
    echo "Please specify the path manually:"
    echo "Usage: $0 /path/to/your/image.img"
    exit 1
fi

echo ""
echo "🔍 Checking IMG file status..."

# Check if file is currently in use
BOCHS_PIDS=$(ps aux | grep bochs | grep -v grep | awk '{print $2}')
if [ ! -z "$BOCHS_PIDS" ]; then
    echo "⚠️  BOCHS is currently running (PIDs: $BOCHS_PIDS)"
    echo "   Killing BOCHS processes..."
    for pid in $BOCHS_PIDS; do
        kill -9 $pid 2>/dev/null
        echo "   Killed process $pid"
    done
    sleep 2
fi

# Check file permissions and ownership
echo ""
echo "📋 File Information:"
ls -lh "$XP_IMAGE"

# Check if file is writable
if [ ! -w "$XP_IMAGE" ]; then
    echo "❌ File is not writable"
    echo "   Attempting to fix permissions..."
    chmod 644 "$XP_IMAGE"
    if [ $? -eq 0 ]; then
        echo "✅ Permissions fixed"
    else
        echo "❌ Failed to fix permissions (may need root access)"
    fi
fi

echo ""
echo "🔧 IMG Unlock Options:"
echo ""
echo "1. Force unlock (remove lock files)"
echo "2. Check for corruption"
echo "3. Create backup before reset"
echo "4. Reset to clean state (if you have backup)"
echo "5. Exit"
echo ""
read -p "Choose option (1-5): " choice

case $choice in
    1)
        echo ""
        echo "🔓 Force unlocking IMG file..."
        
        # Remove any lock files that might exist
        LOCK_FILES=(
            "${XP_IMAGE}.lock"
            "${XP_IMAGE}.lck"
            "${XP_IMAGE}.tmp"
            "/tmp/bochs_${XP_IMAGE##*/}.lock"
        )
        
        for lock in "${LOCK_FILES[@]}"; do
            if [ -f "$lock" ]; then
                rm -f "$lock"
                echo "   Removed lock file: $lock"
            fi
        done
        
        # Clear any shared memory segments
        ipcs -m | grep $(whoami) | awk '{print $2}' | xargs -r ipcrm -m 2>/dev/null
        
        echo "✅ IMG file unlocked"
        echo "   You can now try booting again"
        ;;
        
    2)
        echo ""
        echo "🔍 Checking for corruption..."
        
        # Basic file integrity check
        if [ -r "$XP_IMAGE" ]; then
            SIZE=$(stat -c%s "$XP_IMAGE" 2>/dev/null)
            if [ "$SIZE" -gt 0 ]; then
                echo "✅ File appears intact (size: $(ls -lh "$XP_IMAGE" | awk '{print $5}'))"
                
                # Check if it looks like a valid disk image
                if file "$XP_IMAGE" | grep -q "boot sector"; then
                    echo "✅ Valid boot sector detected"
                else
                    echo "⚠️  No boot sector detected (may be raw image)"
                fi
            else
                echo "❌ File is empty or corrupted"
            fi
        else
            echo "❌ Cannot read file"
        fi
        ;;
        
    3)
        echo ""
        echo "💾 Creating backup..."
        BACKUP_NAME="${XP_IMAGE}.backup.$(date +%Y%m%d_%H%M%S)"
        
        echo "   Copying to: $BACKUP_NAME"
        if cp "$XP_IMAGE" "$BACKUP_NAME"; then
            echo "✅ Backup created successfully"
            echo "   Original size: $(ls -lh "$XP_IMAGE" | awk '{print $5}')"
            echo "   Backup size:   $(ls -lh "$BACKUP_NAME" | awk '{print $5}')"
        else
            echo "❌ Backup failed"
        fi
        ;;
        
    4)
        echo ""
        echo "🔄 Reset to clean state..."
        echo "⚠️  WARNING: This will restore from the most recent backup"
        echo ""
        
        # Find most recent backup
        BACKUP_DIR=$(dirname "$XP_IMAGE")
        BACKUP_FILE=$(ls -t "${XP_IMAGE}.backup."* 2>/dev/null | head -1)
        
        if [ -f "$BACKUP_FILE" ]; then
            echo "   Found backup: $BACKUP_FILE"
            read -p "   Restore from this backup? (y/N): " confirm
            if [ "$confirm" = "y" ] || [ "$confirm" = "Y" ]; then
                if cp "$BACKUP_FILE" "$XP_IMAGE"; then
                    echo "✅ IMG file restored from backup"
                else
                    echo "❌ Restore failed"
                fi
            fi
        else
            echo "❌ No backup files found"
            echo "   Backup files should be named: ${XP_IMAGE}.backup.*"
        fi
        ;;
        
    5)
        echo "Exiting..."
        exit 0
        ;;
        
    *)
        echo "Invalid option"
        exit 1
        ;;
esac

echo ""
echo "=========================================="
echo "IMG unlock/reset complete!"
echo "You can now try running BOCHS again."
echo "=========================================="
