#!/bin/sh
# XSDL Mouse Test and Troubleshooting for BOCHS

# Get the directory where this script is located
SCRIPT_DIR="$(cd "$(dirname "$0")" && pwd)"

# Set library path
export LD_LIBRARY_PATH="$SCRIPT_DIR/lib:/usr/local/lib:/usr/lib:/lib:$LD_LIBRARY_PATH"

# Set BOCHS share path
export BXSHARE="$SCRIPT_DIR/share/bochs"

# Set display to XSDL server
export DISPLAY=192.168.2.2:1

echo "=========================================="
echo "XSDL MOUSE TEST & TROUBLESHOOTING"
echo "=========================================="
echo ""
echo "This will test mouse functionality with XSDL"
echo "and help diagnose input issues."
echo ""

# Test X11 connection first
echo "🔍 Testing X11 connection to XSDL..."
if xset q >/dev/null 2>&1; then
    echo "✅ X11 connection successful"
else
    echo "❌ X11 connection failed"
    echo "   Make sure XSDL is running on 192.168.1.107:0"
    echo "   Try restarting XSDL app"
    exit 1
fi

echo ""
echo "🖱️  XSDL Mouse Configuration Tips:"
echo ""
echo "In XSDL app settings:"
echo "1. Enable 'Mouse Emulation'"
echo "2. Set 'Touch Mode' to 'Mouse'"
echo "3. Enable 'Right Click' (long press)"
echo "4. Disable 'Relative Mouse' if enabled"
echo "5. Set 'Mouse Sensitivity' to medium"
echo ""
echo "BOCHS Mouse Controls:"
echo "- Ctrl+F10 = Release mouse capture"
echo "- Ctrl+F12 = Toggle fullscreen"
echo "- Ctrl+Alt+Del = Send to guest OS"
echo ""

# Create minimal test configuration
cat > "$SCRIPT_DIR/mouse_test.bochsrc" << 'EOFCONFIG'
# Minimal BOCHS configuration for mouse testing

# Basic settings
megs: 64
romimage: file=$BXSHARE/BIOS-bochs-latest
vgaromimage: file=$BXSHARE/VGABIOS-lgpl-latest

# Boot from floppy (no OS needed for mouse test)
boot: floppy
floppya: 1_44=/dev/null, status=ejected

# Display
display_library: x

# CPU
cpu: count=1, ips=1000000, model=bx_generic, reset_on_triple_fault=1

# Clock
clock: sync=none

# MOUSE SETTINGS - Multiple configurations to test
mouse: enabled=1, type=ps2, toggle=ctrl+mbutton

# Keyboard
keyboard: keymap=, user_shortcut=none

# Minimal logging
log: -
panic: action=fatal
error: action=ignore
info: action=ignore
debug: action=ignore

# Display
private_colormap: enabled=0
EOFCONFIG

echo "=== Starting Mouse Test ==="
echo "This will boot to BIOS screen for mouse testing"
echo "Move mouse around and click to test functionality"
echo "Press Ctrl+C in terminal to exit when done"
echo ""

# Run test
"$SCRIPT_DIR/bin/bochs" -q -f "$SCRIPT_DIR/mouse_test.bochsrc"
